/*
* @sum.cumo/vue-datepicker v4.0.0
* (c) 2018-2022 sum.cumo GmbH
* Released under the Apache-2.0 License.
*/
class Language {
  // eslint-disable-next-line max-params
  constructor(language, months, monthsAbbr, days, rtl = false, ymd = false, yearSuffix = '') {
    this.language = language;
    this.months = months;
    this.monthsAbbr = monthsAbbr;
    this.days = days;
    this.rtl = rtl;
    this.ymd = ymd;
    this.yearSuffix = yearSuffix;
  }
  /* eslint-disable no-underscore-dangle */


  get language() {
    return this._language;
  }

  set language(language) {
    if (typeof language !== 'string') {
      throw new TypeError('Language must be a string');
    }

    this._language = language;
  }

  get months() {
    return this._months;
  }

  set months(months) {
    if (months.length !== 12) {
      throw new RangeError(`There must be 12 months for ${this.language} language`);
    }

    this._months = months;
  }

  get monthsAbbr() {
    return this._monthsAbbr;
  }

  set monthsAbbr(monthsAbbr) {
    if (monthsAbbr.length !== 12) {
      throw new RangeError(`There must be 12 abbreviated months for ${this.language} language`);
    }

    this._monthsAbbr = monthsAbbr;
  }

  get days() {
    return this._days;
  }

  set days(days) {
    if (days.length !== 7) {
      throw new RangeError(`There must be 7 days for ${this.language} language`);
    }

    this._days = days;
  }

  getDaysStartingOn(firstDayOfWeek) {
    const firstDays = this._days.slice(firstDayOfWeek);

    const lastDays = this._days.slice(0, firstDayOfWeek);

    return firstDays.concat(lastDays);
  }

  getMonthByAbbrName(name) {
    const monthValue = this._monthsAbbr.findIndex(month => month === name) + 1;
    return monthValue < 10 ? `0${monthValue}` : `${monthValue}`;
  }

  getMonthByName(name) {
    const monthValue = this._months.findIndex(month => month === name) + 1;
    return monthValue < 10 ? `0${monthValue}` : `${monthValue}`;
  }

}

var en = new Language('English', ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'], ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'], ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat']);

var calendarSlots = ['beforeCalendarHeaderDay', 'calendarFooterDay', 'beforeCalendarHeaderMonth', 'calendarFooterMonth', 'beforeCalendarHeaderYear', 'calendarFooterYear', 'nextIntervalBtn', 'prevIntervalBtn'];

const getParsedDate = ({
  formatStr,
  dateStr,
  translation
}) => {
  const splitter = formatStr.match(/-|\/|\s|\./) || ['-'];
  const df = formatStr.split(splitter[0]);
  const ds = dateStr.split(splitter[0]);
  const ymd = [new Date().getFullYear(), '01', '01'];

  for (let i = 0; i < df.length; i += 1) {
    if (/yyyy/i.test(df[i])) {
      ymd[0] = ds[i];
    } else if (/mmmm/i.test(df[i])) {
      ymd[1] = translation.getMonthByName(ds[i]);
    } else if (/mmm/i.test(df[i])) {
      ymd[1] = translation.getMonthByAbbrName(ds[i]);
    } else if (/mm/i.test(df[i])) {
      ymd[1] = ds[i];
    } else if (/m/i.test(df[i])) {
      ymd[1] = ds[i];
    } else if (/dd/i.test(df[i])) {
      ymd[2] = ds[i];
    } else if (/d/i.test(df[i])) {
      const tmp = ds[i].replace(/st|rd|nd|th/g, '');
      ymd[2] = tmp < 10 ? `0${tmp}` : `${tmp}`;
    }
  }

  return ymd;
};

const utils = {
  /**
   * @type {Boolean}
   */
  useUtc: false,

  /**
   * Returns the full year, using UTC or not
   * @param {Date} date
   */
  getFullYear(date) {
    return this.useUtc ? date.getUTCFullYear() : date.getFullYear();
  },

  /**
   * Returns the month, using UTC or not
   * @param {Date} date
   */
  getMonth(date) {
    return this.useUtc ? date.getUTCMonth() : date.getMonth();
  },

  /**
   * Returns the number of days in the month, using UTC or not
   * @param {Date} date
   */
  getDaysInMonth(date) {
    return this.daysInMonth(this.getFullYear(date), this.getMonth(date));
  },

  /**
   * Returns the date, using UTC or not
   * @param {Date} date
   */
  getDate(date) {
    return this.useUtc ? date.getUTCDate() : date.getDate();
  },

  /**
   * Returns the day, using UTC or not
   * @param {Date} date
   */
  getDay(date) {
    return this.useUtc ? date.getUTCDay() : date.getDay();
  },

  /**
   * Returns the hours, using UTC or not
   * @param {Date} date
   */
  getHours(date) {
    return this.useUtc ? date.getUTCHours() : date.getHours();
  },

  /**
   * Returns the minutes, using UTC or not
   * @param {Date} date
   */
  getMinutes(date) {
    return this.useUtc ? date.getUTCMinutes() : date.getMinutes();
  },

  /**
   * Sets the full year, using UTC or not
   * @param {Date} date
   * @param {String, Number} value
   */
  setFullYear(date, value) {
    return this.useUtc ? date.setUTCFullYear(value) : date.setFullYear(value);
  },

  /**
   * Sets the month, using UTC or not
   * @param {Date} date
   * @param {String, Number} value
   */
  setMonth(date, value) {
    return this.useUtc ? date.setUTCMonth(value) : date.setMonth(value);
  },

  /**
   * Sets the date, using UTC or not
   * @param {Date} date
   * @param {String, Number} value
   */
  setDate(date, value) {
    return this.useUtc ? date.setUTCDate(value) : date.setDate(value);
  },

  /**
   * Check if date1 is equivalent to date2, without comparing the time
   * @see https://stackoverflow.com/a/6202196/4455925
   * @param {Date} date1
   * @param {Date} date2
   */
  compareDates(date1, date2) {
    const d1 = new Date(date1.valueOf());
    const d2 = new Date(date2.valueOf());
    this.resetDateTime(d1);
    this.resetDateTime(d2);
    return d1.valueOf() === d2.valueOf();
  },

  /**
   * Validates a date object
   * @param {Date} date - an object instantiated with the new Date constructor
   * @return {Boolean}
   */
  isValidDate(date) {
    if (Object.prototype.toString.call(date) !== '[object Date]') {
      return false;
    }

    return !Number.isNaN(date.valueOf());
  },

  /**
   * Return abbreviated week day name
   * @param {Date} date
   * @param {Array} days
   * @return {String}
   */
  getDayNameAbbr(date, days) {
    if (typeof date !== 'object') {
      throw TypeError('Invalid Type');
    }

    return days[this.getDay(date)];
  },

  /**
   * Return day number from abbreviated week day name
   * @param {String} abbr
   * @return {Number}
   */
  getDayFromAbbr(abbr) {
    for (let i = 0; i < en.days.length; i += 1) {
      if (abbr.toLowerCase() === en.days[i].toLowerCase()) {
        return i;
      }
    }

    throw TypeError('Invalid week day');
  },

  /**
   * Return name of the month
   * @param {Number|Date} month
   * @param {Array} months
   * @return {String}
   */
  getMonthName(month, months) {
    if (!months) {
      throw Error('missing 2nd parameter Months array');
    }

    if (typeof month === 'object') {
      return months[this.getMonth(month)];
    }

    if (typeof month === 'number') {
      return months[month];
    }

    throw TypeError('Invalid type');
  },

  /**
   * Return an abbreviated version of the month
   * @param {Number|Date} month
   * @param {Array} monthsAbbr
   * @return {String}
   */
  getMonthNameAbbr(month, monthsAbbr) {
    if (!monthsAbbr) {
      throw Error('missing 2nd parameter Months array');
    }

    if (typeof month === 'object') {
      return monthsAbbr[this.getMonth(month)];
    }

    if (typeof month === 'number') {
      return monthsAbbr[month];
    }

    throw TypeError('Invalid type');
  },

  /**
   * Alternative get total number of days in month
   * @param {Number} year
   * @param {Number} month
   * @return {Number}
   */
  // eslint-disable-next-line complexity
  daysInMonth(year, month) {
    if (/8|3|5|10/.test(month.toString())) {
      return 30;
    }

    if (month === 1) {
      return !(year % 4) && year % 100 || !(year % 400) ? 29 : 28;
    }

    return 31;
  },

  /**
   * Get nth suffix for date
   * @param {Number} day
   * @return {String}
   */
  // eslint-disable-next-line complexity
  getNthSuffix(day) {
    switch (day) {
      case 1:
      case 21:
      case 31:
        return 'st';

      case 2:
      case 22:
        return 'nd';

      case 3:
      case 23:
        return 'rd';

      default:
        return 'th';
    }
  },

  /**
   * Formats date object
   * @param {Date} date
   * @param {String} formatStr
   * @param {Object} translation
   * @return {String}
   */
  formatDate(date, formatStr, translation = en) {
    const year = this.getFullYear(date);
    const month = this.getMonth(date) + 1;
    const day = this.getDate(date);
    const matches = {
      dd: `0${day}`.slice(-2),
      d: day,
      yyyy: year,
      yy: String(year).slice(2),
      MMMM: this.getMonthName(this.getMonth(date), translation.months),
      MMM: this.getMonthNameAbbr(this.getMonth(date), translation.monthsAbbr),
      MM: `0${month}`.slice(-2),
      M: month,
      o: this.getNthSuffix(this.getDate(date)),
      E: this.getDayNameAbbr(date, translation.days)
    };
    const REGEX_FORMAT = /y{4}|y{2}|M{1,4}(?![aäe])|d{1,2}|o|E(?![eéi])/g;
    return formatStr.replace(REGEX_FORMAT, match => matches[match] || match);
  },

  /**
   * makes date parseable
   * to use with international dates
   * @param {String} dateStr
   * @param {String|Function} formatStr
   * @param {Object} translation
   * @param {Function} parser
   * @return {Date | String}
   */
  // eslint-disable-next-line max-params,complexity,max-statements
  parseDate(dateStr, formatStr, translation = en, parser = null) {
    if (!(dateStr && formatStr)) {
      return dateStr;
    }

    if (typeof formatStr === 'function') {
      if (!parser || typeof parser !== 'function') {
        throw new Error('Parser needs to be a function if you are using a custom formatter');
      }

      return parser(dateStr);
    }

    const ymd = getParsedDate({
      formatStr,
      dateStr,
      translation
    });
    const dat = `${ymd.join('-')}${this.getTime()}`;

    if (Number.isNaN(Date.parse(dat))) {
      return dateStr;
    }

    return dat;
  },

  getTime() {
    const time = 'T00:00:00';

    if (this.useUtc) {
      return `${time}Z`;
    }

    return time;
  },

  /**
   * Creates an array of dates for each day in between two dates.
   * @param {Date} start
   * @param {Date} end
   * @return {Array}
   */
  createDateArray(start, end) {
    const dates = [];
    let startTemp = start;

    while (startTemp <= end) {
      dates.push(new Date(startTemp));
      startTemp = this.setDate(new Date(startTemp), this.getDate(new Date(startTemp)) + 1);
    }

    return dates;
  },

  /**
   * Remove hours/minutes/seconds/milliseconds from a date object
   * @param {Date} date
   * @return {Date}
   */
  resetDateTime(date) {
    return new Date(this.useUtc ? date.setUTCHours(0, 0, 0, 0) : date.setHours(0, 0, 0, 0));
  },

  /**
   * Return a new date object with hours/minutes/seconds/milliseconds removed.
   * Defaults to today's date, if no parameter is provided
   * @param {Date=} date
   * @return {Date}
   */
  getNewDateObject(date) {
    return date ? this.resetDateTime(new Date(date)) : this.resetDateTime(new Date());
  }

};
var makeDateUtils = (useUtc => ({ ...utils,
  useUtc
}));

var script$b = {
  props: {
    autofocus: {
      type: Boolean,
      default: false
    },
    bootstrapStyling: {
      type: Boolean,
      default: false
    },
    clearButton: {
      type: Boolean,
      default: false
    },
    clearButtonIcon: {
      type: String,
      default: ''
    },
    calendarButton: {
      type: Boolean,
      default: false
    },
    calendarButtonIcon: {
      type: String,
      default: ''
    },
    calendarButtonIconContent: {
      type: String,
      default: ''
    },
    disabled: {
      type: Boolean,
      default: false
    },
    format: {
      type: [String, Function],
      default: 'dd MMM yyyy'
    },
    id: {
      type: String,
      default: null
    },
    inline: {
      type: Boolean,
      default: false
    },
    inputClass: {
      type: [String, Object, Array],
      default: null
    },
    maxlength: {
      type: [Number, String],
      default: null
    },
    name: {
      type: String,
      default: null
    },
    openDate: {
      type: [String, Date, Number],
      default: null,
      validator: val => val === null || val instanceof Date || typeof val === 'string' || typeof val === 'number'
    },
    parser: {
      type: Function,
      default: null
    },
    pattern: {
      type: String,
      default: null
    },
    placeholder: {
      type: String,
      default: null
    },
    refName: {
      type: String,
      default: ''
    },
    required: {
      type: Boolean,
      default: false
    },
    showCalendarOnButtonClick: {
      type: Boolean,
      default: false
    },
    showCalendarOnFocus: {
      type: Boolean,
      default: false
    },
    tabindex: {
      type: [Number, String],
      default: null
    },
    typeable: {
      type: Boolean,
      default: false
    },
    useUtc: {
      type: Boolean,
      default: false
    }
  }
};

function normalizeComponent(template, style, script, scopeId, isFunctionalTemplate, moduleIdentifier /* server only */, shadowMode, createInjector, createInjectorSSR, createInjectorShadow) {
    if (typeof shadowMode !== 'boolean') {
        createInjectorSSR = createInjector;
        createInjector = shadowMode;
        shadowMode = false;
    }
    // Vue.extend constructor export interop.
    const options = typeof script === 'function' ? script.options : script;
    // render functions
    if (template && template.render) {
        options.render = template.render;
        options.staticRenderFns = template.staticRenderFns;
        options._compiled = true;
        // functional template
        if (isFunctionalTemplate) {
            options.functional = true;
        }
    }
    // scopedId
    if (scopeId) {
        options._scopeId = scopeId;
    }
    let hook;
    if (moduleIdentifier) {
        // server build
        hook = function (context) {
            // 2.3 injection
            context =
                context || // cached call
                    (this.$vnode && this.$vnode.ssrContext) || // stateful
                    (this.parent && this.parent.$vnode && this.parent.$vnode.ssrContext); // functional
            // 2.2 with runInNewContext: true
            if (!context && typeof __VUE_SSR_CONTEXT__ !== 'undefined') {
                context = __VUE_SSR_CONTEXT__;
            }
            // inject component styles
            if (style) {
                style.call(this, createInjectorSSR(context));
            }
            // register component module identifier for async chunk inference
            if (context && context._registeredComponents) {
                context._registeredComponents.add(moduleIdentifier);
            }
        };
        // used by ssr in case component is cached and beforeCreate
        // never gets called
        options._ssrRegister = hook;
    }
    else if (style) {
        hook = shadowMode
            ? function (context) {
                style.call(this, createInjectorShadow(context, this.$root.$options.shadowRoot));
            }
            : function (context) {
                style.call(this, createInjector(context));
            };
    }
    if (hook) {
        if (options.functional) {
            // register for functional component in vue file
            const originalRender = options.render;
            options.render = function renderWithStyleInjection(h, context) {
                hook.call(context);
                return originalRender(h, context);
            };
        }
        else {
            // inject component registration as beforeCreate hook
            const existing = options.beforeCreate;
            options.beforeCreate = existing ? [].concat(existing, hook) : [hook];
        }
    }
    return script;
}

/* script */
const __vue_script__$b = script$b;

/* template */

  /* style */
  const __vue_inject_styles__$b = undefined;
  /* scoped */
  const __vue_scope_id__$b = undefined;
  /* module identifier */
  const __vue_module_identifier__$b = undefined;
  /* functional template */
  const __vue_is_functional_template__$b = undefined;
  /* style inject */
  
  /* style inject SSR */
  
  /* style inject shadow dom */
  

  
  const __vue_component__$b = /*#__PURE__*/normalizeComponent(
    {},
    __vue_inject_styles__$b,
    __vue_script__$b,
    __vue_scope_id__$b,
    __vue_is_functional_template__$b,
    __vue_module_identifier__$b,
    false,
    undefined,
    undefined,
    undefined
  );

//
var script$a = {
  name: 'DateInput',
  mixins: [__vue_component__$b],
  props: {
    isOpen: {
      type: Boolean,
      default: false
    },
    selectedDate: {
      type: Date,
      default: null
    },
    translation: {
      type: Object,

      default() {
        return {};
      }

    }
  },

  data() {
    return {
      input: null,
      isInputFocused: false,
      shouldToggleOnFocus: false,
      shouldToggleOnClick: true,
      parsedDate: null,
      typedDate: '',
      utils: makeDateUtils(this.useUtc)
    };
  },

  computed: {
    computedInputClass() {
      if (this.bootstrapStyling) {
        if (typeof this.inputClass === 'string') {
          return [this.inputClass, 'form-control'].join(' ');
        }

        return {
          'form-control': true,
          ...this.inputClass
        };
      }

      return this.inputClass;
    },

    formattedDate() {
      if (!this.selectedDate) {
        return null;
      }

      return typeof this.format === 'function' ? this.format(new Date(this.selectedDate)) : this.utils.formatDate(new Date(this.selectedDate), this.format, this.translation);
    },

    formattedValue() {
      if (!this.selectedDate) {
        return null;
      }

      if (this.typedDate.length) {
        return this.typedDate;
      }

      return this.formattedDate;
    }

  },
  watch: {
    showCalendarOnFocus: {
      immediate: true,

      handler(showCalendarOnFocus) {
        if (showCalendarOnFocus) {
          this.shouldToggleOnFocus = !this.isOpen;
        }
      }

    },

    isOpen(isOpen, wasOpen) {
      this.$nextTick(() => {
        if (this.showCalendarOnFocus) {
          if (isOpen) {
            this.shouldToggleOnFocus = false;
          }

          if (wasOpen && !this.isInputFocused) {
            this.shouldToggleOnFocus = true;
          }
        }
      });
    }

  },

  mounted() {
    this.input = this.$el.querySelector('input');
  },

  methods: {
    /**
     * Emits a `clear-date` event
     */
    clearDate() {
      this.input.value = '';
      this.$emit('clear-date');
    },

    /**
     * Formats a typed date, or clears it if invalid
     */
    formatTypedDate() {
      if (Number.isNaN(this.parsedDate)) {
        this.input.value = '';
        this.typedDate = '';
      } else {
        this.typedDate = this.formattedDate;
      }
    },

    /**
     * Validate typedDate and emit a `blur` event
     */
    handleInputBlur() {
      if (this.showCalendarOnFocus && !this.isOpen) {
        this.shouldToggleOnFocus = true;
      }

      if (this.typeable) {
        this.formatTypedDate();
      }

      this.isInputFocused = false;
      this.$emit('blur');
    },

    /**
     * Resets `shouldToggleOnFocus` to true
     */
    handleButtonFocus() {
      if (this.showCalendarOnFocus) {
        this.shouldToggleOnFocus = true;
      }
    },

    /**
     * Clears the calendar when the `delete` or `backspace` key is pressed
     */
    handleDelete() {
      if (!this.typeable && this.selectedDate) {
        this.clearDate();
      }
    },

    /**
     * Toggles the calendar (unless `show-calendar-on-button-click` is true)
     */
    handleInputClick() {
      if (this.showCalendarOnButtonClick) return;

      if (this.shouldToggleOnClick) {
        this.toggle();
      }
    },

    /**
     * Emits a `focus` event and opens the calendar when `show-calendar-on-focus` is true
     */
    handleInputFocus() {
      this.isInputFocused = true;

      if (!this.isOpen && this.shouldToggleOnFocus) {
        this.shouldToggleOnClick = false;
      }

      if (this.shouldToggleOnFocus && !this.isOpen) {
        this.$emit('open');
        setTimeout(() => {
          this.shouldToggleOnClick = true;
        }, 300);
      }

      this.$emit('focus');
    },

    /**
     * Opens the calendar, or sets the focus to the next focusable element down
     */
    handleKeydownDown() {
      if (!this.isOpen) {
        this.$emit('open');
      }

      if (!this.typeable) {
        return;
      }

      this.$emit('set-focus', ['prev', 'up', 'next', 'tabbableCell']);
    },

    /**
     * Formats a typed date and closes the calendar
     */
    handleKeydownEnter() {
      if (!this.typeable) {
        return;
      }

      this.formatTypedDate();

      if (this.isOpen) {
        this.$emit('close');
      }
    },

    /**
     * Closes the calendar
     */
    handleKeydownEscape() {
      if (this.isOpen) {
        this.$emit('close');
      }
    },

    /**
     * Prevents scrolling when not typeable
     */
    handleKeydownSpace(event) {
      if (!this.typeable) {
        event.preventDefault();
      }
    },

    /**
     * Parses a typed date and submits it, if valid
     * @param  {object}  event Used to exclude certain keystrokes
     */
    handleKeyup(event) {
      if (!this.typeable || ['Shift', 'Tab', 'ArrowUp', 'ArrowDown', 'ArrowLeft', 'ArrowRight'].includes(event.key)) {
        return;
      }

      if (this.input.value === '') {
        this.$emit('typed-date', null);
        return;
      }

      this.parsedDate = Date.parse(this.utils.parseDate(this.input.value, this.format, this.translation, this.parser));

      if (!Number.isNaN(this.parsedDate)) {
        this.typedDate = this.input.value;
        this.$emit('typed-date', new Date(this.parsedDate));
      }
    },

    /**
     * Toggles the calendar unless a typed date has been entered or `show-calendar-on-button-click` is true
     */
    handleKeyupSpace(event) {
      if (this.typeable) {
        if (this.input.value === '') {
          this.toggle();
        }

        return;
      }

      event.preventDefault();

      if (!this.showCalendarOnButtonClick) {
        this.toggle();
      }
    },

    /**
     * Opens or closes the calendar
     */
    toggle(calendarButton) {
      if (this.isOpen) {
        this.$emit('set-focus', [calendarButton || 'input']);
      }

      this.$emit(this.isOpen ? 'close' : 'open');
    }

  }
};

/* script */
const __vue_script__$a = script$a;

/* template */
var __vue_render__$7 = function () {
  var _vm = this;
  var _h = _vm.$createElement;
  var _c = _vm._self._c || _h;
  return _c(
    "div",
    { class: { "input-group": _vm.bootstrapStyling } },
    [
      _vm._t("beforeDateInput"),
      _vm._v(" "),
      _vm.calendarButton
        ? _c(
            "button",
            {
              ref: "calendarButton",
              staticClass: "vdp-datepicker__calendar-button",
              class: { "btn input-group-prepend": _vm.bootstrapStyling },
              attrs: {
                "data-test-calendar-button": "",
                disabled: _vm.disabled,
                type: "button",
              },
              on: {
                click: function ($event) {
                  return _vm.toggle("calendarButton")
                },
                focus: _vm.handleButtonFocus,
              },
            },
            [
              _c(
                "span",
                { class: { "input-group-text": _vm.bootstrapStyling } },
                [
                  _vm._t("calendarBtn", [
                    _c("i", { class: _vm.calendarButtonIcon }, [
                      _vm._v(
                        "\n          " +
                          _vm._s(_vm.calendarButtonIconContent) +
                          "\n          "
                      ),
                      !_vm.calendarButtonIcon
                        ? _c("span", [_vm._v("…")])
                        : _vm._e(),
                    ]),
                  ]),
                ],
                2
              ),
            ]
          )
        : _vm._e(),
      _vm._v(" "),
      _c("input", {
        ref: _vm.refName,
        class: _vm.computedInputClass,
        attrs: {
          id: _vm.id,
          autocomplete: "off",
          autofocus: _vm.autofocus,
          "clear-button": _vm.clearButton,
          "data-test-input": "",
          disabled: _vm.disabled,
          maxlength: _vm.maxlength,
          name: _vm.name,
          pattern: _vm.pattern,
          placeholder: _vm.placeholder,
          readonly: !_vm.typeable,
          required: _vm.required,
          tabindex: _vm.tabindex,
          type: _vm.inline ? "hidden" : null,
        },
        domProps: { value: _vm.formattedValue },
        on: {
          blur: _vm.handleInputBlur,
          click: _vm.handleInputClick,
          focus: _vm.handleInputFocus,
          keydown: [
            function ($event) {
              if (
                !$event.type.indexOf("key") &&
                _vm._k(
                  $event.keyCode,
                  "backspace",
                  undefined,
                  $event.key,
                  undefined
                )
              ) {
                return null
              }
              return _vm.handleDelete($event)
            },
            function ($event) {
              if (
                !$event.type.indexOf("key") &&
                _vm._k($event.keyCode, "delete", [8, 46], $event.key, [
                  "Backspace",
                  "Delete",
                  "Del",
                ])
              ) {
                return null
              }
              return _vm.handleDelete($event)
            },
            function ($event) {
              if (
                !$event.type.indexOf("key") &&
                _vm._k($event.keyCode, "down", 40, $event.key, [
                  "Down",
                  "ArrowDown",
                ])
              ) {
                return null
              }
              $event.preventDefault();
              return _vm.handleKeydownDown($event)
            },
            function ($event) {
              if (
                !$event.type.indexOf("key") &&
                _vm._k($event.keyCode, "enter", 13, $event.key, "Enter")
              ) {
                return null
              }
              $event.preventDefault();
              return _vm.handleKeydownEnter($event)
            },
            function ($event) {
              if (
                !$event.type.indexOf("key") &&
                _vm._k($event.keyCode, "esc", 27, $event.key, ["Esc", "Escape"])
              ) {
                return null
              }
              $event.preventDefault();
              return _vm.handleKeydownEscape($event)
            },
            function ($event) {
              if (
                !$event.type.indexOf("key") &&
                _vm._k($event.keyCode, "space", 32, $event.key, [
                  " ",
                  "Spacebar",
                ])
              ) {
                return null
              }
              return _vm.handleKeydownSpace($event)
            },
          ],
          keyup: [
            function ($event) {
              return _vm.handleKeyup($event)
            },
            function ($event) {
              if (
                !$event.type.indexOf("key") &&
                _vm._k($event.keyCode, "space", 32, $event.key, [
                  " ",
                  "Spacebar",
                ])
              ) {
                return null
              }
              return _vm.handleKeyupSpace($event)
            },
          ],
        },
      }),
      _vm._v(" "),
      _vm.clearButton && _vm.selectedDate
        ? _c(
            "button",
            {
              staticClass: "vdp-datepicker__clear-button",
              class: { "btn input-group-append": _vm.bootstrapStyling },
              attrs: {
                "data-test-clear-button": "",
                disabled: _vm.disabled,
                type: "button",
              },
              on: { click: _vm.clearDate },
            },
            [
              _c(
                "span",
                { class: { "input-group-text": _vm.bootstrapStyling } },
                [
                  _vm._t("clearBtn", [
                    _c("i", { class: _vm.clearButtonIcon }, [
                      !_vm.clearButtonIcon
                        ? _c("span", [_vm._v("×")])
                        : _vm._e(),
                    ]),
                  ]),
                ],
                2
              ),
            ]
          )
        : _vm._e(),
      _vm._v(" "),
      _vm._t("afterDateInput"),
    ],
    2
  )
};
var __vue_staticRenderFns__$7 = [];
__vue_render__$7._withStripped = true;

  /* style */
  const __vue_inject_styles__$a = undefined;
  /* scoped */
  const __vue_scope_id__$a = undefined;
  /* module identifier */
  const __vue_module_identifier__$a = undefined;
  /* functional template */
  const __vue_is_functional_template__$a = false;
  /* style inject */
  
  /* style inject SSR */
  
  /* style inject shadow dom */
  

  
  const __vue_component__$a = /*#__PURE__*/normalizeComponent(
    { render: __vue_render__$7, staticRenderFns: __vue_staticRenderFns__$7 },
    __vue_inject_styles__$a,
    __vue_script__$a,
    __vue_scope_id__$a,
    __vue_is_functional_template__$a,
    __vue_module_identifier__$a,
    false,
    undefined,
    undefined,
    undefined
  );

const cellUtils = {
  configExists(obj) {
    return typeof obj !== 'undefined' && Object.keys(obj).length > 0;
  },

  isDefined(obj, prop) {
    return this.configExists(obj) && typeof obj[prop] !== 'undefined';
  },

  hasArray(obj, prop) {
    return this.isDefined(obj, prop) && obj[prop].length > 0;
  },

  hasDate(obj, prop) {
    return this.isDefined(obj, prop) && this.utils.isValidDate(obj[prop]);
  },

  dayMonthYear(obj, prop) {
    const {
      utils
    } = this;
    const hasDate = this.hasDate(obj, prop);

    if (!hasDate) {
      return {
        day: undefined,
        month: undefined,
        year: undefined
      };
    }

    const d = obj[prop];
    return {
      day: utils.getDate(d),
      month: utils.getMonth(d),
      year: utils.getFullYear(d)
    };
  }

};
var makeCellUtils = (utils => ({ ...cellUtils,
  utils
}));

/* eslint-disable no-underscore-dangle */
class DisabledDate {
  constructor(utils, disabledDates) {
    this._utils = utils;
    this._disabledDates = disabledDates;
  }

  get config() {
    const disabledDates = this._disabledDates;
    const utils = makeCellUtils(this._utils);
    return {
      exists: utils.configExists(disabledDates),
      to: utils.dayMonthYear(disabledDates, 'to'),
      from: utils.dayMonthYear(disabledDates, 'from'),
      has: {
        customPredictor: utils.isDefined(disabledDates, 'customPredictor'),
        daysOfMonth: utils.hasArray(disabledDates, 'daysOfMonth'),
        daysOfWeek: utils.hasArray(disabledDates, 'days'),
        from: utils.hasDate(disabledDates, 'from'),
        ranges: utils.hasArray(disabledDates, 'ranges'),
        specificDates: utils.hasArray(disabledDates, 'dates'),
        to: utils.hasDate(disabledDates, 'to')
      }
    };
  }

  daysInMonth(date) {
    const utils = this._utils;
    const month = utils.getMonth(date);
    const year = utils.getFullYear(date);
    return utils.daysInMonth(year, month);
  }

  isDateDisabledVia(date) {
    const disabledDates = this._disabledDates;
    const {
      has
    } = this.config;
    return {
      to: () => {
        return has.to && date < disabledDates.to;
      },
      from: () => {
        return has.from && date > disabledDates.from;
      },
      range: () => {
        if (!has.ranges) return false;
        const {
          ranges
        } = disabledDates;
        const u = makeCellUtils(this._utils);
        return ranges.some(thisRange => {
          const hasFrom = u.isDefined(thisRange, 'from');
          const hasTo = u.isDefined(thisRange, 'to');
          return hasFrom && hasTo && date < thisRange.to && date > thisRange.from;
        });
      },
      customPredictor: () => {
        return has.customPredictor && disabledDates.customPredictor(date);
      },
      specificDate: () => {
        if (!has.specificDates) return false;
        return disabledDates.dates.some(d => {
          return this._utils.compareDates(date, d);
        });
      },
      daysOfWeek: () => {
        if (!has.daysOfWeek) return false;
        return disabledDates.days.indexOf(this._utils.getDay(date)) !== -1;
      },
      daysOfMonth: () => {
        if (!has.daysOfMonth) return false;
        return disabledDates.daysOfMonth.indexOf(this._utils.getDate(date)) !== -1;
      }
    };
  }

  isMonthDisabledVia(date) {
    const {
      from,
      has,
      to
    } = this.config;

    const month = this._utils.getMonth(date);

    const year = this._utils.getFullYear(date);

    return {
      to: () => {
        const isYearInPast = has.to && year < to.year;

        if (isYearInPast) {
          return true;
        }

        return has.to && month < to.month && year <= to.year;
      },
      from: () => {
        const isYearInFuture = has.from && year > from.year;

        if (isYearInFuture) {
          return true;
        }

        return has.from && month > from.month && year >= from.year;
      }
    };
  }

  isYearDisabledVia(date) {
    const {
      from,
      has,
      to
    } = this.config;

    const year = this._utils.getFullYear(date);

    return {
      to: () => {
        return has.to && year < to.year;
      },
      from: () => {
        return has.from && year > from.year;
      }
    };
  }
  /**
   * Checks if the given date should be disabled
   * @param {Date} date
   * @return {Boolean}
   */
  // eslint-disable-next-line complexity,max-statements


  isDateDisabled(date) {
    if (!this.config.exists) return false;
    const isDisabledVia = this.isDateDisabledVia(date);
    return isDisabledVia.to() || isDisabledVia.from() || isDisabledVia.range() || isDisabledVia.specificDate() || isDisabledVia.daysOfWeek() || isDisabledVia.daysOfMonth() || isDisabledVia.customPredictor();
  }
  /**
   * Checks if the given month should be disabled
   * @param {Date} date
   * @return {Boolean}
   */
  // eslint-disable-next-line complexity,max-statements


  isMonthDisabled(date) {
    const {
      config
    } = this;
    const isDisabledVia = this.isMonthDisabledVia(date);

    if (!config.exists) {
      return false;
    }

    if (isDisabledVia.to() || isDisabledVia.from()) {
      return true;
    } // now we have to check each day of the month


    for (let i = 1; i <= this.daysInMonth(date); i += 1) {
      const dayDate = new Date(date);
      dayDate.setDate(i); // if at least one day of this month is NOT disabled,
      // we can conclude that this month SHOULD be selectable

      if (!this.isDateDisabled(dayDate)) {
        return false;
      }
    }

    return true;
  }
  /**
   * Checks if the given year should be disabled
   * @param {Date} date
   * @return {Boolean}
   */
  // eslint-disable-next-line complexity,max-statements


  isYearDisabled(date) {
    const {
      config
    } = this;
    const isDisabledVia = this.isYearDisabledVia(date);

    if (!config.exists) {
      return false;
    }

    if (isDisabledVia.to() || isDisabledVia.from()) {
      return true;
    } // now we have to check each month of the year


    for (let i = 0; i < 12; i += 1) {
      const monthDate = new Date(date);
      monthDate.setMonth(i); // if at least one month of this year is NOT disabled,
      // we can conclude that this year SHOULD be selectable

      if (!this.isMonthDisabled(monthDate)) {
        return false;
      }
    }

    return true;
  }

  getEarliestPossibleDate(date) {
    if (!date) {
      return null;
    }

    const utils = this._utils;

    if (this.isDateDisabled(date)) {
      const nextDate = new Date(utils.getFullYear(date), utils.getMonth(date), utils.getDate(date) + 1);
      return this.getEarliestPossibleDate(nextDate);
    }

    return date;
  }

  getLatestPossibleDate(date) {
    if (!date) {
      return null;
    }

    const utils = this._utils;

    if (this.isDateDisabled(date)) {
      const nextDate = new Date(utils.getFullYear(date), utils.getMonth(date), utils.getDate(date) - 1);
      return this.getLatestPossibleDate(nextDate);
    }

    return date;
  }

}

var script$9 = {
  data() {
    return {
      allElements: [],
      focus: {
        delay: 0,
        refs: []
      },
      inlineTabbableCell: null,
      isActive: false,
      isRevertingToOpenDate: false,
      navElements: [],
      navElementsFocusedIndex: 0,
      resetTabbableCell: false,
      tabbableCell: null,
      transitionName: ''
    };
  },

  computed: {
    fallbackElementsToFocus() {
      const elements = ['tabbableCell', 'prev', 'next'];

      if (this.typeable) {
        elements.unshift('input');
      }

      return elements;
    },

    focusedDateTimestamp() {
      const pageDate = new Date(this.pageTimestamp);

      if (this.hasClass(this.tabbableCell, 'day')) {
        return this.utils.setDate(pageDate, this.tabbableCell.innerHTML.trim());
      }

      if (this.hasClass(this.tabbableCell, 'month')) {
        return this.utils.setMonth(pageDate, this.tabbableCellId);
      }

      const fullYear = this.utils.getFullYear(pageDate) - 1;
      return this.utils.setFullYear(pageDate, fullYear + this.tabbableCellId);
    },

    tabbableCellId() {
      return this.tabbableCell && Number(this.tabbableCell.getAttribute('data-id'));
    }

  },
  methods: {
    /**
     * Returns true, unless tabbing should be focus-trapped
     * @return {Boolean}
     */
    allowNormalTabbing(event) {
      if (!this.isOpen) {
        return true;
      }

      return this.isTabbingAwayFromInlineDatepicker(event);
    },

    /**
     * Focuses the first non-disabled element found in the `focus.refs` array and sets `navElementsFocusedIndex`
     */
    applyFocus() {
      const focusRefs = [...this.focus.refs, ...this.fallbackElementsToFocus];

      for (let i = 0; i < focusRefs.length; i += 1) {
        const element = this.getElementByRef(focusRefs[i]);

        if (element && !element.getAttribute('disabled')) {
          element.focus();
          this.setNavElementsFocusedIndex();
          break;
        }
      }
    },

    /**
     * Ensures the most recently focused tabbable cell is focused when tabbing backwards to an inline calendar
     * If no element has previously been focused, the tabbable cell is reset and focused
     */
    focusInlineTabbableCell() {
      if (this.inlineTabbableCell) {
        this.inlineTabbableCell.focus();
        return;
      }

      this.resetTabbableCell = true;
      this.setTabbableCell();
      this.tabbableCell.focus();
      this.resetTabbableCell = false;
    },

    /**
     * Returns the currently focused cell element, if there is one...
     */
    getActiveCell() {
      const isActiveElementACell = this.hasClass(document.activeElement, 'cell');
      const isOnSameView = this.hasClass(document.activeElement, this.view);

      if (isActiveElementACell && isOnSameView && !this.resetTabbableCell) {
        return document.activeElement;
      }

      return null;
    },

    /**
     * Finds an element by its `ref` attribute
     * @param {string} ref        The `ref` name of the wanted element
     * @returns {HTMLElement|Vue} A Vue element
     */
    // eslint-disable-next-line complexity,max-statements
    getElementByRef(ref) {
      if (ref === 'tabbableCell') {
        return this.tabbableCell;
      }

      if (ref === 'input') {
        return this.$refs.dateInput && this.$refs.dateInput.$refs[this.refName];
      }

      if (ref === 'calendarButton') {
        return this.$refs.dateInput.$refs.calendarButton;
      }

      if (ref === 'openDate') {
        return this.$refs.picker.$refs.cells.$refs.openDate[0];
      }

      if (this.showHeader) {
        if (ref === 'up') {
          return this.$refs.picker && this.$refs.picker.$refs.up.$el;
        }

        return this.$refs.picker && this.$refs.picker.$refs.pickerHeader && this.$refs.picker.$refs.pickerHeader.$refs[ref];
      }

      return null;
    },

    /**
     * Returns an array of all HTML elements which should be focus-trapped in the calendarFooter slot
     * @returns {Array}   An array of HTML elements
     */
    getElementsFromCalendarFooter() {
      const footerSlotIndex = this.hasSlot('beforeCalendarHeader') ? 2 : 1;
      return this.getFocusableElements(this.$refs.view.children[footerSlotIndex]);
    },

    /**
     * Returns an array of all HTML elements which should be focus-trapped in the specified slot
     * @returns {Array}   An array of HTML elements
     */
    getElementsFromSlot(slotName) {
      if (!this.hasSlot(slotName)) {
        return [];
      }

      if (slotName === 'beforeCalendarHeader') {
        return this.getFocusableElements(this.$refs.view.children[0]);
      }

      if (slotName === 'calendarFooter') {
        return this.getElementsFromCalendarFooter();
      }

      const isBeforeHeader = slotName.indexOf('beforeCalendarHeader') > -1;
      const picker = this.$refs.picker.$el;
      const index = isBeforeHeader ? 0 : picker.children.length - 1;
      return this.getFocusableElements(picker.children[index]);
    },

    /**
     * Returns an array of all HTML elements which should be focus-trapped in the header
     * @returns {Array}   An array of HTML elements
     */
    getElementsFromHeader() {
      const view = this.ucFirst(this.view);
      const beforeCalendarSlotName = `beforeCalendarHeader${view}`;
      const picker = this.$refs.picker.$el;
      const index = this.hasSlot(beforeCalendarSlotName) ? 1 : 0;
      const fragment = picker.children[index];
      return this.showHeader ? this.getFocusableElements(fragment) : [];
    },

    /**
     * Returns an array of focusable elements in a given HTML fragment
     * @param   {Element} fragment The HTML fragment to search
     * @returns {Array}
     */
    getFocusableElements(fragment) {
      const navNodeList = fragment.querySelectorAll('button:enabled, [href], input:not([type=hidden]), select:enabled, textarea:enabled, [tabindex]:not([tabindex="-1"])');
      return [...Array.prototype.slice.call(navNodeList)];
    },

    /**
     * Returns the input element (when typeable)
     * @returns {Element}
     */
    getInputField() {
      if (!this.typeable || this.inline) {
        return null;
      }

      return this.$refs.dateInput.$refs[this.refName];
    },

    /**
     * Sets `datepickerId` (as a global) and keeps track of focusable elements
     */
    handleFocusIn() {
      document.datepickerId = this.datepickerId;
      this.isActive = true;
      this.setInlineTabbableCell();
      this.setAllElements();
      this.setNavElements();
    },

    /**
     * Sets the datepicker's `isActive` state to false
     */
    handleFocusOut() {
      this.isActive = false;
    },

    /**
     * Returns true if the user has arrowed to a new page
     * @return {Boolean}
     */
    hasArrowedToNewPage() {
      return this.focus.refs && this.focus.refs[0] === 'arrow-to-cell';
    },

    /**
     * Returns true if the calendar has been passed the given slot
     * @param  {String} slotName The name of the slot
     * @return {Boolean}
     */
    hasSlot(slotName) {
      return !!this.$slots[slotName];
    },

    /**
     * Returns true if the user is tabbing away from an inline datepicker
     * @return {Boolean}
     */
    isTabbingAwayFromInlineDatepicker(event) {
      if (!this.inline) {
        return false;
      }

      if (this.isTabbingAwayFromFirstNavElement(event)) {
        this.tabAwayFromFirstElement();
        return true;
      }

      if (this.isTabbingAwayFromLastNavElement(event)) {
        this.tabAwayFromLastElement();
        return true;
      }

      return false;
    },

    /**
     * Used for inline calendars; returns true if the user tabs backwards from the first focusable element
     * @param  {object}  event Used to determine whether we are tabbing forwards or backwards
     * @return {Boolean}
     */
    isTabbingAwayFromFirstNavElement(event) {
      if (!event.shiftKey) {
        return false;
      }

      const firstNavElement = this.navElements[0];
      return document.activeElement === firstNavElement;
    },

    /**
     * Used for inline calendars; returns true if the user tabs forwards from the last focusable element
     * @param  {object}  event Used to determine whether we are tabbing forwards or backwards
     * @return {Boolean}
     */
    isTabbingAwayFromLastNavElement(event) {
      if (event.shiftKey) {
        return false;
      }

      const lastNavElement = this.navElements[this.navElements.length - 1];
      return document.activeElement === lastNavElement;
    },

    /**
     * Resets the focus to the open date
     */
    resetFocusToOpenDate() {
      this.focus.refs = ['openDate'];
      this.setTransitionAndFocusDelay(this.focusedDateTimestamp, this.computedOpenDate);

      if (!this.isMinimumView) {
        this.isRevertingToOpenDate = true;
        this.view = this.minimumView;
      }

      this.setPageDate(this.computedOpenDate);
      this.reviewFocus();
    },

    /**
     * Sets the correct focus on next tick
     */
    reviewFocus() {
      if (this.hasArrowedToNewPage()) {
        return;
      }

      this.tabbableCell = null;
      this.resetTabbableCell = true;
      this.$nextTick(() => {
        this.setNavElements();
        setTimeout(() => {
          this.applyFocus();
        }, this.focus.delay);
        this.resetTabbableCell = false;
      });
    },

    /**
     * Stores the current tabbableCell of an inline datepicker
     * N.B. This is used when tabbing back (shift + tab) to an inline calendar from further down the page
     */
    setInlineTabbableCell() {
      if (!this.inline) {
        return;
      }

      this.inlineTabbableCell = this.tabbableCell;
    },

    /**
     * Sets the direction of the slide transition and whether or not to delay application of the focus
     * @param {Date|Number} startDate     The date from which to measure
     * @param {Date|Number} endDate       Is this before or after the startDate? And is it on the same page?
     */
    setTransitionAndFocusDelay(startDate, endDate) {
      const startPageDate = this.utils.setDate(new Date(startDate), 1);
      const endPageDate = this.utils.setDate(new Date(endDate), 1);
      const isInTheFuture = startPageDate < endPageDate;

      if (this.isMinimumView) {
        this.focus.delay = isInTheFuture ? this.slideDuration : 0;
      } else {
        this.focus.delay = 0;
      }

      this.setTransitionName(endDate - startDate);
    },

    /**
     * Records all focusable elements (so that we know whether any element in the datepicker is focused)
     */
    setAllElements() {
      this.allElements = this.getFocusableElements(this.$refs.datepicker);

      if (this.appendToBody) {
        this.allElements = this.allElements.concat(this.getFocusableElements(this.$refs.popup.$el));
      }
    },

    /**
     * Set the focus
     * @param {Array} refs An array of `refs` to focus (in order of preference)
     */
    setFocus(refs) {
      this.focus.refs = refs;
      this.applyFocus();
    },

    /**
     * Determines which elements in datepicker should be focus-trapped
     */
    setNavElements() {
      if (!this.view) return;
      this.updateTabbableCell();
      const view = this.ucFirst(this.view);
      this.navElements = [this.getElementsFromSlot('beforeDateInput'), this.getInputField(), this.getElementsFromSlot('afterDateInput'), this.getElementsFromSlot('beforeCalendarHeader'), this.getElementsFromSlot(`beforeCalendarHeader${view}`), this.getElementsFromHeader(), this.tabbableCell, this.getElementsFromSlot(`calendarFooter${view}`), this.getElementsFromSlot('calendarFooter')].filter(item => !!item).reduce((acc, val) => acc.concat(val), []);
    },

    /**
     * Keeps track of the currently focused index in the navElements array
     */
    setNavElementsFocusedIndex() {
      for (let i = 0; i < this.navElements.length; i += 1) {
        if (document.activeElement === this.navElements[i]) {
          this.navElementsFocusedIndex = i;
          return;
        }
      }

      this.navElementsFocusedIndex = 0;
    },

    /**
     * Sets the focus-trapped cell in the picker
     */
    // eslint-disable-next-line complexity
    setTabbableCell() {
      if (!this.$refs.picker || !this.$refs.picker.$refs.cells) {
        return;
      }

      const pickerCells = this.$refs.picker.$refs.cells.$el;
      this.tabbableCell = this.getActiveCell() || pickerCells.querySelector('button.selected:not(.muted):enabled') || pickerCells.querySelector('button.open:not(.muted):enabled') || pickerCells.querySelector('button.today:not(.muted):enabled') || pickerCells.querySelector('button.cell:not(.muted):enabled');
    },

    /**
     * Sets the direction of the slide transition
     * @param {Number} plusOrMinus Positive for the future; negative for the past
     */
    setTransitionName(plusOrMinus) {
      const isInTheFuture = plusOrMinus > 0;

      if (this.isRtl) {
        this.transitionName = isInTheFuture ? 'slide-left' : 'slide-right';
      } else {
        this.transitionName = isInTheFuture ? 'slide-right' : 'slide-left';
      }
    },

    /**
     * Focuses the first focusable element in the datepicker, so that the previous element on the page will be tabbed to
     */
    tabAwayFromFirstElement() {
      const firstElement = this.allElements[0];
      firstElement.focus();
      this.tabbableCell = this.inlineTabbableCell;
    },

    /**
     * Focuses the last focusable element in the datepicker, so that the next element on the page will be tabbed to
     */
    tabAwayFromLastElement() {
      const lastElement = this.allElements[this.allElements.length - 1];
      lastElement.focus();
      this.tabbableCell = this.inlineTabbableCell;
    },

    /**
     * Tab backwards through the focus-trapped elements
     */
    tabBackwards() {
      this.navElementsFocusedIndex -= 1;

      if (this.navElementsFocusedIndex < 0) {
        this.navElementsFocusedIndex = this.navElements.length - 1;
      }

      this.navElements[this.navElementsFocusedIndex].focus();
    },

    /**
     * Tab forwards through the focus-trapped elements
     */
    tabForwards() {
      this.navElementsFocusedIndex += 1;

      if (this.navElementsFocusedIndex >= this.navElements.length) {
        this.navElementsFocusedIndex = 0;
      }

      this.navElements[this.navElementsFocusedIndex].focus();
    },

    /**
     * Tab through the focus-trapped elements
     * @param event
     */
    tabThroughNavigation(event) {
      if (this.allowNormalTabbing(event)) {
        return;
      }

      event.preventDefault();

      if (event.shiftKey) {
        this.tabBackwards();
      } else {
        this.tabForwards();
      }
    },

    /**
     * Special cases for when tabbing to an inline datepicker
     */
    tabToCorrectInlineCell() {
      const lastElement = this.allElements[this.allElements.length - 1];
      const isACell = this.hasClass(lastElement, 'cell');
      const isLastElementFocused = document.activeElement === lastElement; // If there are no focusable elements in the footer slots and the inline datepicker has been tabbed to (backwards)

      if (isACell && isLastElementFocused) {
        this.focusInlineTabbableCell();
        return;
      } // If `show-header` is false and the inline datepicker has been tabbed to (forwards)


      this.$nextTick(() => {
        const isFirstCell = document.activeElement.getAttribute('data-id') === '0';

        if (isFirstCell) {
          this.focusInlineTabbableCell();
        }
      });
    },

    /**
     * Update which cell in the picker should be focus-trapped
     */
    updateTabbableCell() {
      const isActiveElementACell = this.hasClass(document.activeElement, 'cell');
      const needToUpdate = !this.tabbableCell || isActiveElementACell;

      if (needToUpdate) {
        this.setTabbableCell();
      }
    }

  }
};

/* script */
const __vue_script__$9 = script$9;

/* template */

  /* style */
  const __vue_inject_styles__$9 = undefined;
  /* scoped */
  const __vue_scope_id__$9 = undefined;
  /* module identifier */
  const __vue_module_identifier__$9 = undefined;
  /* functional template */
  const __vue_is_functional_template__$9 = undefined;
  /* style inject */
  
  /* style inject SSR */
  
  /* style inject shadow dom */
  

  
  const __vue_component__$9 = /*#__PURE__*/normalizeComponent(
    {},
    __vue_inject_styles__$9,
    __vue_script__$9,
    __vue_scope_id__$9,
    __vue_is_functional_template__$9,
    __vue_module_identifier__$9,
    false,
    undefined,
    undefined,
    undefined
  );

//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
var script$8 = {
  name: 'PickerHeader',
  props: {
    bootstrapStyling: {
      type: Boolean,
      default: false
    },
    isNextDisabled: {
      type: Boolean,
      required: true
    },
    isPreviousDisabled: {
      type: Boolean,
      required: true
    },
    isRtl: {
      type: Boolean,
      required: true
    }
  },

  data() {
    return {
      previousPage: {
        incrementBy: -1,
        focusRefs: ['prev']
      },
      nextPage: {
        incrementBy: 1,
        focusRefs: ['next']
      }
    };
  },

  methods: {
    /**
     * Changes the page, or sets focus to the adjacent button
     */
    arrowLeftPrev() {
      if (this.isRtl) {
        this.$emit('set-focus', ['up', 'next', 'tabbableCell']);
        return;
      }

      this.$emit('page-change', this.previousPage);
    },

    /**
     * Changes the page, or sets focus to the adjacent button
     */
    arrowRightPrev() {
      if (this.isRtl) {
        this.$emit('page-change', this.previousPage);
        return;
      }

      this.$emit('set-focus', ['up', 'next', 'tabbableCell']);
    },

    /**
     * Changes the page, or sets focus to the adjacent button
     */
    arrowLeftNext() {
      if (this.isRtl) {
        this.$emit('page-change', this.nextPage);
        return;
      }

      this.$emit('set-focus', ['up', 'prev', 'tabbableCell']);
    },

    /**
     * Changes the page, or sets focus to the adjacent button
     */
    arrowRightNext() {
      if (this.isRtl) {
        this.$emit('set-focus', ['up', 'prev', 'tabbableCell']);
        return;
      }

      this.$emit('page-change', this.nextPage);
    },

    focusTabbableCell() {
      this.$emit('set-focus', ['tabbableCell']);
    }

  }
};

/* script */
const __vue_script__$8 = script$8;

/* template */
var __vue_render__$6 = function () {
  var _vm = this;
  var _h = _vm.$createElement;
  var _c = _vm._self._c || _h;
  return _c(
    "header",
    [
      _c(
        "button",
        {
          ref: "prev",
          staticClass: "prev",
          class: { btn: _vm.bootstrapStyling, rtl: _vm.isRtl },
          attrs: {
            "data-test-previous-button": "",
            disabled: _vm.isPreviousDisabled,
            type: "button",
          },
          on: {
            click: function ($event) {
              $event.stopPropagation();
              return _vm.$emit("page-change", _vm.previousPage)
            },
            keydown: [
              function ($event) {
                if (
                  !$event.type.indexOf("key") &&
                  _vm._k($event.keyCode, "down", 40, $event.key, [
                    "Down",
                    "ArrowDown",
                  ])
                ) {
                  return null
                }
                $event.preventDefault();
                return _vm.focusTabbableCell($event)
              },
              function ($event) {
                if (
                  !$event.type.indexOf("key") &&
                  _vm._k($event.keyCode, "up", 38, $event.key, [
                    "Up",
                    "ArrowUp",
                  ])
                ) {
                  return null
                }
                $event.preventDefault();
                return _vm.$emit("focus-input")
              },
              function ($event) {
                if (
                  !$event.type.indexOf("key") &&
                  _vm._k($event.keyCode, "left", 37, $event.key, [
                    "Left",
                    "ArrowLeft",
                  ])
                ) {
                  return null
                }
                if ("button" in $event && $event.button !== 0) {
                  return null
                }
                $event.preventDefault();
                return _vm.arrowLeftPrev($event)
              },
              function ($event) {
                if (
                  !$event.type.indexOf("key") &&
                  _vm._k($event.keyCode, "right", 39, $event.key, [
                    "Right",
                    "ArrowRight",
                  ])
                ) {
                  return null
                }
                if ("button" in $event && $event.button !== 2) {
                  return null
                }
                $event.preventDefault();
                return _vm.arrowRightPrev($event)
              },
            ],
          },
        },
        [
          _vm._t("prevIntervalBtn", [
            _c("span", { staticClass: "default" }, [_vm._v("<")]),
          ]),
        ],
        2
      ),
      _vm._v(" "),
      _vm._t("default"),
      _vm._v(" "),
      _c(
        "button",
        {
          ref: "next",
          staticClass: "next",
          class: { btn: _vm.bootstrapStyling, rtl: _vm.isRtl },
          attrs: {
            "data-test-next-button": "",
            disabled: _vm.isNextDisabled,
            type: "button",
          },
          on: {
            click: function ($event) {
              $event.stopPropagation();
              return _vm.$emit("page-change", _vm.nextPage)
            },
            keydown: [
              function ($event) {
                if (
                  !$event.type.indexOf("key") &&
                  _vm._k($event.keyCode, "down", 40, $event.key, [
                    "Down",
                    "ArrowDown",
                  ])
                ) {
                  return null
                }
                $event.preventDefault();
                return _vm.focusTabbableCell($event)
              },
              function ($event) {
                if (
                  !$event.type.indexOf("key") &&
                  _vm._k($event.keyCode, "up", 38, $event.key, [
                    "Up",
                    "ArrowUp",
                  ])
                ) {
                  return null
                }
                $event.preventDefault();
                return _vm.$emit("focus-input")
              },
              function ($event) {
                if (
                  !$event.type.indexOf("key") &&
                  _vm._k($event.keyCode, "left", 37, $event.key, [
                    "Left",
                    "ArrowLeft",
                  ])
                ) {
                  return null
                }
                if ("button" in $event && $event.button !== 0) {
                  return null
                }
                $event.preventDefault();
                return _vm.arrowLeftNext($event)
              },
              function ($event) {
                if (
                  !$event.type.indexOf("key") &&
                  _vm._k($event.keyCode, "right", 39, $event.key, [
                    "Right",
                    "ArrowRight",
                  ])
                ) {
                  return null
                }
                if ("button" in $event && $event.button !== 2) {
                  return null
                }
                $event.preventDefault();
                return _vm.arrowRightNext($event)
              },
            ],
          },
        },
        [
          _vm._t("nextIntervalBtn", [
            _c("span", { staticClass: "default" }, [_vm._v(">")]),
          ]),
        ],
        2
      ),
    ],
    2
  )
};
var __vue_staticRenderFns__$6 = [];
__vue_render__$6._withStripped = true;

  /* style */
  const __vue_inject_styles__$8 = undefined;
  /* scoped */
  const __vue_scope_id__$8 = undefined;
  /* module identifier */
  const __vue_module_identifier__$8 = undefined;
  /* functional template */
  const __vue_is_functional_template__$8 = false;
  /* style inject */
  
  /* style inject SSR */
  
  /* style inject shadow dom */
  

  
  const __vue_component__$8 = /*#__PURE__*/normalizeComponent(
    { render: __vue_render__$6, staticRenderFns: __vue_staticRenderFns__$6 },
    __vue_inject_styles__$8,
    __vue_script__$8,
    __vue_scope_id__$8,
    __vue_is_functional_template__$8,
    __vue_module_identifier__$8,
    false,
    undefined,
    undefined,
    undefined
  );

var script$7 = {
  components: {
    PickerHeader: __vue_component__$8
  },
  inheritAttrs: false,
  props: {
    bootstrapStyling: {
      type: Boolean,
      default: false
    },
    disabledDates: {
      type: Object,

      default() {
        return {};
      }

    },
    isRtl: {
      type: Boolean,
      default: false
    },
    isTypeable: {
      type: Boolean,
      default: false
    },
    isUpDisabled: {
      type: Boolean,
      default: false
    },
    isMinimumView: {
      type: Boolean,
      default: true
    },
    openDate: {
      type: [String, Date, Number],
      default: null,
      validator: val => val === null || val instanceof Date || typeof val === 'string' || typeof val === 'number'
    },
    pageDate: {
      type: Date,
      default: null
    },
    selectedDate: {
      type: Date,
      default: null
    },
    showHeader: {
      type: Boolean,
      default: true
    },
    slideDuration: {
      type: Number,
      default: 250
    },
    tabbableCellId: {
      type: Number,
      default: null
    },
    transitionName: {
      type: String,
      default: ''
    },
    translation: {
      type: Object,

      default() {
        return {};
      }

    },
    useUtc: {
      type: Boolean,
      default: false
    },
    view: {
      type: String,
      default: 'day'
    }
  },

  data() {
    return {
      utils: makeDateUtils(this.useUtc)
    };
  },

  computed: {
    /**
     * A look-up object created from 'disabledDates' prop
     * @return {Object}
     */
    disabledConfig() {
      return new DisabledDate(this.utils, this.disabledDates).config;
    },

    earliestPossibleDate() {
      return new DisabledDate(this.utils, this.disabledDates).getEarliestPossibleDate(this.disabledDates.to);
    },

    latestPossibleDate() {
      return new DisabledDate(this.utils, this.disabledDates).getLatestPossibleDate(this.disabledDates.from);
    },

    /**
     * Returns the current page's full year as an integer.
     * @return {Number}
     */
    pageYear() {
      return this.utils.getFullYear(this.pageDate);
    }

  },
  methods: {
    /**
     * Used when an arrow key press would cause the focus to land on a disabled date
     * @param {Object} options
     */
    addMoreSteps(options) {
      if (options.stepsRemaining <= 0 && Math.abs(options.delta) > 1) {
        return Math.abs(options.delta);
      }

      return options.stepsRemaining;
    },

    /**
     * Changes the page up or down
     * @param {Number} incrementBy
     * @param {[String]} focusRefs
     */
    changePage({
      incrementBy,
      focusRefs
    }) {
      const {
        pageDate,
        utils
      } = this;
      const units = this.view === 'year' ? incrementBy * this.yearRange : incrementBy;
      this.$emit('set-transition-name', incrementBy);

      if (this.view === 'day') {
        utils.setMonth(pageDate, utils.getMonth(pageDate) + units);
      } else {
        utils.setFullYear(pageDate, utils.getFullYear(pageDate) + units);
      }

      this.$emit('page-change', {
        focusRefs,
        pageDate
      });
    },

    /**
     * Changes the page and focuses the cell that is being 'arrowed' to
     * @param {Object} options
     */
    changePageAndSetFocus(options) {
      const {
        delta
      } = options;
      const isPageDisabled = delta > 0 && this.isNextDisabled || delta < 0 && this.isPreviousDisabled;

      if (isPageDisabled) {
        return;
      }

      this.changePage({
        incrementBy: Math.sign(delta),
        focusRefs: ['arrow-to-cell']
      });
      this.$nextTick(() => {
        this.setFocusOnNewPage(options);
      });
    },

    /**
     * Focuses the input field, if typeable
     */
    focusInput() {
      if (this.isTypeable) {
        this.$emit('set-focus', ['input']);
      }
    },

    /**
     * Returns the element that should be focused when navigating via an arrow key
     * @param  {HTMLButtonElement} currentElement  The element currently being iterated on
     * @param  {Number}            delta           The number of cells that the focus should move
     * @param  {Number}            stepsRemaining  The number of steps remaining in the iteration
     * @return {HTMLButtonElement | void}
     */
    // eslint-disable-next-line complexity,max-statements
    getElement({
      currentElement,
      delta,
      stepsRemaining
    }) {
      const element = this.getElementSibling(currentElement, delta);
      const options = {
        currentElement: element,
        delta,
        stepsRemaining: stepsRemaining - 1
      };

      if (!element) {
        return this.changePageAndSetFocus(options);
      }

      if (this.isBeyondPossibleDate(options)) {
        return this.firstOrLastPossibleDate(options);
      }

      if (this.isMutedOrDisabled(element)) {
        options.stepsRemaining = this.addMoreSteps(options);
        return this.getElement(options);
      }

      if (stepsRemaining > 1 && options.currentElement) {
        return this.getElement(options);
      }

      return element;
    },

    /**
     * Returns the element directly next to the currentElement
     * @param  {HTMLButtonElement} currentElement The element currently being iterated on
     * @param  {Number}            delta          The number of cells that the focus should move
     * @return {HTMLButtonElement}
     */
    getElementSibling(currentElement, delta) {
      const isNext = delta > 0;
      return isNext ? currentElement.nextElementSibling : currentElement.previousElementSibling;
    },

    /**
     * Returns the first or last cell, depending on the direction of the search
     * @param  {Number} delta The number of cells that the focus should move
     * @return {HTMLButtonElement}
     */
    getFirstOrLastElement(delta) {
      const isNext = delta > 0;
      const elements = this.$refs.cells.$el.children;
      return isNext ? elements[0] : elements[elements.length - 1];
    },

    /**
     * Returns the first or last non-disabled date, depending on the direction of the search
     * @param  {HTMLButtonElement} currentElement  The element currently being iterated on
     * @param  {Number}            delta           The number of cells that the focus should move
     */
    firstOrLastPossibleDate({
      currentElement,
      delta
    }) {
      if (delta > 0) {
        return this.getElementSibling(currentElement, -1);
      }

      return this.getElementSibling(currentElement, 1);
    },

    /**
     * Moves the focused cell up/down/left/right
     * @param {Object}
     */
    handleArrow({
      delta
    }) {
      const stepsRemaining = Math.abs(delta);
      const options = {
        currentElement: document.activeElement,
        delta,
        stepsRemaining
      };
      this.setFocusToAvailableCell(options);
    },

    /**
     * Determines which transition to use (for edge dates) and emits a 'select' event
     * @param {Object} cell
     */
    select(cell) {
      if (cell.isPreviousMonth) {
        this.$emit('set-transition-name', -1);
      }

      if (cell.isNextMonth) {
        this.$emit('set-transition-name', 1);
      }

      this.$emit('select', cell);
    },

    /**
     * Returns true if the given element cannot be focused
     * @param  {HTMLButtonElement} currentElement  The element currently being iterated on
     * @param  {Number}            delta           The number of cells that the focus should move
     * @return {Boolean}
     */
    isBeyondPossibleDate({
      currentElement,
      delta
    }) {
      if (delta > 0 && this.latestPossibleDate) {
        return this.isDatePossible(currentElement, delta);
      }

      if (delta < 0 && this.earliestPossibleDate) {
        return this.isDatePossible(currentElement, delta);
      }

      return false;
    },

    /**
     * Returns true if the current element's date is NOT possible, given the `disabled-dates`
     * @param  {HTMLButtonElement} element The element in question
     * @param  {Number}            delta   Used to determine direction of travel
     * @return {Boolean}
     */
    isDatePossible(element, delta) {
      const cellId = element.getAttribute('data-id');
      const cellDate = new Date(this.cells[cellId].timestamp);

      if (delta > 0) {
        return cellDate > this.latestPossibleDate;
      }

      return cellDate < this.earliestPossibleDate;
    },

    /**
     * Returns true if the given element cannot be focused
     * @param  {HTMLButtonElement} element The element in question
     * @return {Boolean}
     */
    isMutedOrDisabled(element) {
      const isMuted = element.classList.value.split(' ').includes('muted');
      const isDisabled = element.disabled;
      return isMuted || isDisabled;
    },

    /**
     * Sets the focus on the correct cell following a page change
     * @param {Object} options
     */
    setFocusOnNewPage({
      delta,
      stepsRemaining
    }) {
      const currentElement = this.getFirstOrLastElement(delta);
      const options = {
        currentElement,
        delta,
        stepsRemaining
      };

      if (stepsRemaining <= 0) {
        if (this.isMutedOrDisabled(currentElement)) {
          options.stepsRemaining = Math.abs(options.delta);
          setTimeout(() => {
            this.setFocusToAvailableCell(options);
          }, this.slideDuration);
          return;
        }

        setTimeout(() => {
          currentElement.focus();
        }, this.slideDuration);
        return;
      }

      setTimeout(() => {
        this.setFocusToAvailableCell(options);
      }, this.slideDuration);
    },

    /**
     * Sets the focus on the next focusable cell when an arrow key is pressed
     * @param {Object} options
     */
    setFocusToAvailableCell(options) {
      const element = this.getElement(options);

      if (element) {
        element.focus();
      }
    }

  }
};

/* script */
const __vue_script__$7 = script$7;

/* template */

  /* style */
  const __vue_inject_styles__$7 = undefined;
  /* scoped */
  const __vue_scope_id__$7 = undefined;
  /* module identifier */
  const __vue_module_identifier__$7 = undefined;
  /* functional template */
  const __vue_is_functional_template__$7 = undefined;
  /* style inject */
  
  /* style inject SSR */
  
  /* style inject shadow dom */
  

  
  const __vue_component__$7 = /*#__PURE__*/normalizeComponent(
    {},
    __vue_inject_styles__$7,
    __vue_script__$7,
    __vue_scope_id__$7,
    __vue_is_functional_template__$7,
    __vue_module_identifier__$7,
    false,
    undefined,
    undefined,
    undefined
  );

/* eslint-disable no-underscore-dangle */
class HighlightedDate {
  constructor(utils, disabledDates, highlighted) {
    this._utils = utils;
    this._disabledDates = disabledDates;
    this._highlighted = highlighted;
  }

  get config() {
    const highlightedDates = this._highlighted;
    const utils = makeCellUtils(this._utils);
    return {
      exists: utils.configExists(highlightedDates),
      to: utils.dayMonthYear(highlightedDates, 'to'),
      from: utils.dayMonthYear(highlightedDates, 'from'),
      has: {
        customPredictor: utils.isDefined(highlightedDates, 'customPredictor'),
        daysOfMonth: utils.hasArray(highlightedDates, 'daysOfMonth'),
        daysOfWeek: utils.hasArray(highlightedDates, 'days'),
        from: utils.hasDate(highlightedDates, 'from'),
        specificDates: utils.hasArray(highlightedDates, 'dates'),
        to: utils.hasDate(highlightedDates, 'to'),
        includeDisabled: utils.isDefined(highlightedDates, 'includeDisabled') && highlightedDates.includeDisabled
      }
    };
  }

  isDateDisabled(date) {
    const utils = this._utils;
    const disabledDates = this._disabledDates;
    return new DisabledDate(utils, disabledDates).isDateDisabled(date);
  }

  isHighlightingNotPossible(date) {
    const {
      config
    } = this;
    if (!config.exists) return false;
    return !config.has.includeDisabled && this.isDateDisabled(date);
  }

  isDateHighlightedVia(date) {
    const highlightedDates = this._highlighted;
    const {
      has
    } = this.config;
    return {
      to: () => {
        return has.to && date <= highlightedDates.to;
      },
      from: () => {
        return has.from && date >= highlightedDates.from;
      },
      customPredictor: () => {
        return has.customPredictor && highlightedDates.customPredictor(date);
      },
      specificDate: () => {
        if (!has.specificDates) return false;
        return highlightedDates.dates.some(d => {
          return this._utils.compareDates(date, d);
        });
      },
      daysOfWeek: () => {
        if (!has.daysOfWeek) return false;
        return highlightedDates.days.indexOf(this._utils.getDay(date)) !== -1;
      },
      daysOfMonth: () => {
        if (!has.daysOfMonth) return false;
        return highlightedDates.daysOfMonth.indexOf(this._utils.getDate(date)) !== -1;
      }
    };
  } // eslint-disable-next-line complexity,max-statements


  isDateHighlighted(date) {
    if (this.isHighlightingNotPossible(date)) return false;
    const isHighlightedVia = this.isDateHighlightedVia(date);
    return isHighlightedVia.to() && isHighlightedVia.from() || isHighlightedVia.specificDate() || isHighlightedVia.daysOfWeek() || isHighlightedVia.daysOfMonth() || isHighlightedVia.customPredictor();
  }

}

//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
var script$6 = {
  name: 'PickerCells',
  props: {
    bootstrapStyling: {
      type: Boolean,
      default: false
    },
    cells: {
      type: Array,
      required: true
    },
    isRtl: {
      type: Boolean,
      default: false
    },
    showEdgeDates: {
      type: Boolean,
      default: true
    },
    tabbableCellId: {
      type: Number,
      default: null
    },
    view: {
      type: String,
      validator: val => ['day', 'month', 'year'].includes(val),
      required: true
    }
  },
  computed: {
    /**
     * The number of columns in the picker
     * @return {Number}
     */
    columns() {
      return this.view === 'day' ? 7 : 3;
    }

  },
  methods: {
    /**
     * Set the classes for a specific cell
     * @return {Array}
     */
    // eslint-disable-next-line complexity
    cellClasses(cell) {
      return ['cell', this.view, {
        'btn': this.bootstrapStyling,
        'disabled': cell.isDisabled,
        'highlight-start': cell.isHighlightStart,
        'highlight-end': cell.isHighlightEnd,
        'highlighted': cell.isHighlighted,
        'muted': cell.isPreviousMonth || cell.isNextMonth,
        'open': cell.isOpenDate,
        'sat': cell.isSaturday,
        'sun': cell.isSunday,
        'selected': this.showEdgeDates ? cell.isSelected : cell.isSelected && !cell.isPreviousMonth && !cell.isNextMonth,
        'today': this.showEdgeDates ? cell.isToday : cell.isToday && !cell.isPreviousMonth && !cell.isNextMonth,
        'weekend': cell.isWeekend
      }];
    },

    /**
     * Emits an `arrow` event
     */
    handleArrow(cellId, delta) {
      this.$emit('arrow', {
        cellId,
        delta
      });
    }

  }
};

/* script */
const __vue_script__$6 = script$6;

/* template */
var __vue_render__$5 = function () {
  var _vm = this;
  var _h = _vm.$createElement;
  var _c = _vm._self._c || _h;
  return _c(
    "div",
    { staticClass: "picker-cells", attrs: { "data-test-picker-cells": "" } },
    _vm._l(_vm.cells, function (cell, id) {
      return _c(
        "button",
        {
          key: cell.timestamp,
          ref: cell.isOpenDate ? "openDate" : null,
          refInFor: true,
          class: _vm.cellClasses(cell),
          attrs: {
            "data-id": id,
            "data-test-tabbable-cell": id === _vm.tabbableCellId,
            "data-test-open-date": cell.isOpenDate,
            "data-test-today-cell": cell.isToday,
            disabled: cell.isDisabled,
            type: "button",
          },
          on: {
            click: function ($event) {
              return _vm.$emit("select", cell)
            },
            keydown: [
              function ($event) {
                if (
                  !$event.type.indexOf("key") &&
                  _vm._k($event.keyCode, "up", 38, $event.key, [
                    "Up",
                    "ArrowUp",
                  ])
                ) {
                  return null
                }
                $event.preventDefault();
                return _vm.handleArrow(id, -_vm.columns)
              },
              function ($event) {
                if (
                  !$event.type.indexOf("key") &&
                  _vm._k($event.keyCode, "down", 40, $event.key, [
                    "Down",
                    "ArrowDown",
                  ])
                ) {
                  return null
                }
                $event.preventDefault();
                return _vm.handleArrow(id, _vm.columns)
              },
              function ($event) {
                if (
                  !$event.type.indexOf("key") &&
                  _vm._k($event.keyCode, "left", 37, $event.key, [
                    "Left",
                    "ArrowLeft",
                  ])
                ) {
                  return null
                }
                if ("button" in $event && $event.button !== 0) {
                  return null
                }
                $event.preventDefault();
                return _vm.handleArrow(id, _vm.isRtl ? 1 : -1)
              },
              function ($event) {
                if (
                  !$event.type.indexOf("key") &&
                  _vm._k($event.keyCode, "right", 39, $event.key, [
                    "Right",
                    "ArrowRight",
                  ])
                ) {
                  return null
                }
                if ("button" in $event && $event.button !== 2) {
                  return null
                }
                $event.preventDefault();
                return _vm.handleArrow(id, _vm.isRtl ? -1 : 1)
              },
            ],
          },
        },
        [_vm._t("default", null, { cell: cell })],
        2
      )
    }),
    0
  )
};
var __vue_staticRenderFns__$5 = [];
__vue_render__$5._withStripped = true;

  /* style */
  const __vue_inject_styles__$6 = undefined;
  /* scoped */
  const __vue_scope_id__$6 = undefined;
  /* module identifier */
  const __vue_module_identifier__$6 = undefined;
  /* functional template */
  const __vue_is_functional_template__$6 = false;
  /* style inject */
  
  /* style inject SSR */
  
  /* style inject shadow dom */
  

  
  const __vue_component__$6 = /*#__PURE__*/normalizeComponent(
    { render: __vue_render__$5, staticRenderFns: __vue_staticRenderFns__$5 },
    __vue_inject_styles__$6,
    __vue_script__$6,
    __vue_scope_id__$6,
    __vue_is_functional_template__$6,
    __vue_module_identifier__$6,
    false,
    undefined,
    undefined,
    undefined
  );

//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
//
var script$5 = {
  name: 'UpButton',
  props: {
    isDisabled: {
      type: Boolean,
      default: false
    },
    isRtl: {
      type: Boolean,
      required: true
    }
  }
};

/* script */
const __vue_script__$5 = script$5;

/* template */
var __vue_render__$4 = function () {
  var _vm = this;
  var _h = _vm.$createElement;
  var _c = _vm._self._c || _h;
  return _c(
    "button",
    {
      staticClass: "vdp-datepicker__up",
      attrs: {
        "data-test-up-button": "",
        disabled: _vm.isDisabled,
        type: "button",
      },
      on: {
        click: function ($event) {
          return _vm.$emit("select")
        },
        keydown: [
          function ($event) {
            if (
              !$event.type.indexOf("key") &&
              _vm._k($event.keyCode, "down", 40, $event.key, [
                "Down",
                "ArrowDown",
              ])
            ) {
              return null
            }
            $event.preventDefault();
            return _vm.$emit("set-focus", ["tabbableCell"])
          },
          function ($event) {
            if (
              !$event.type.indexOf("key") &&
              _vm._k($event.keyCode, "up", 38, $event.key, ["Up", "ArrowUp"])
            ) {
              return null
            }
            $event.preventDefault();
            return _vm.$emit("focus-input")
          },
          function ($event) {
            if (
              !$event.type.indexOf("key") &&
              _vm._k($event.keyCode, "left", 37, $event.key, [
                "Left",
                "ArrowLeft",
              ])
            ) {
              return null
            }
            if ("button" in $event && $event.button !== 0) {
              return null
            }
            $event.preventDefault();
            return _vm.$emit("set-focus", [_vm.isRtl ? "next" : "prev"])
          },
          function ($event) {
            if (
              !$event.type.indexOf("key") &&
              _vm._k($event.keyCode, "right", 39, $event.key, [
                "Right",
                "ArrowRight",
              ])
            ) {
              return null
            }
            if ("button" in $event && $event.button !== 2) {
              return null
            }
            $event.preventDefault();
            return _vm.$emit("set-focus", [_vm.isRtl ? "prev" : "next"])
          },
        ],
      },
    },
    [_vm._t("default")],
    2
  )
};
var __vue_staticRenderFns__$4 = [];
__vue_render__$4._withStripped = true;

  /* style */
  const __vue_inject_styles__$5 = undefined;
  /* scoped */
  const __vue_scope_id__$5 = undefined;
  /* module identifier */
  const __vue_module_identifier__$5 = undefined;
  /* functional template */
  const __vue_is_functional_template__$5 = false;
  /* style inject */
  
  /* style inject SSR */
  
  /* style inject shadow dom */
  

  
  const __vue_component__$5 = /*#__PURE__*/normalizeComponent(
    { render: __vue_render__$4, staticRenderFns: __vue_staticRenderFns__$4 },
    __vue_inject_styles__$5,
    __vue_script__$5,
    __vue_scope_id__$5,
    __vue_is_functional_template__$5,
    __vue_module_identifier__$5,
    false,
    undefined,
    undefined,
    undefined
  );

//
var script$4 = {
  name: 'PickerDay',
  components: {
    PickerCells: __vue_component__$6,
    UpButton: __vue_component__$5
  },
  mixins: [__vue_component__$7],
  props: {
    dayCellContent: {
      type: Function,
      default: day => day.date
    },
    firstDayOfWeek: {
      type: String,
      default: 'sun'
    },
    highlighted: {
      type: Object,

      default() {
        return {};
      }

    },
    showFullMonthName: {
      type: Boolean,
      default: false
    },
    showEdgeDates: {
      type: Boolean,
      default: true
    }
  },
  computed: {
    /**
     * Sets an array with all days to show this month
     * @return {Array}
     */
    cells() {
      const days = [];
      const daysInCalendar = this.daysFromPrevMonth + this.daysInMonth + this.daysFromNextMonth;
      const dObj = this.firstCellDate();

      for (let i = 0; i < daysInCalendar; i += 1) {
        days.push(this.makeDay(i, dObj));
        this.utils.setDate(dObj, this.utils.getDate(dObj) + 1);
      }

      return days;
    },

    /**
     * Gets the name of the month the current page is on
     * @return {String}
     */
    currMonthName() {
      const monthName = this.showFullMonthName ? this.translation.months : this.translation.monthsAbbr;
      return this.utils.getMonthNameAbbr(this.pageMonth, monthName);
    },

    /**
     * Gets the name of the year that current page is on
     * @return {String}
     */
    currYearName() {
      const {
        yearSuffix
      } = this.translation;
      return `${this.pageYear}${yearSuffix}`;
    },

    /**
     * Returns an array of day names
     * @return {String[]}
     */
    daysOfWeek() {
      return this.translation.getDaysStartingOn(this.firstDayOfWeekNumber);
    },

    /**
     * Returns the number of days in this month
     * @return {String[]}
     */
    daysInMonth() {
      return this.utils.getDaysInMonth(this.pageDate);
    },

    /**
     * Calculates how many days to show from the previous month
     * @return {number}
     */
    daysFromPrevMonth() {
      const firstOfMonthDayNumber = this.utils.getDay(this.pageDate);
      return (7 - this.firstDayOfWeekNumber + firstOfMonthDayNumber) % 7;
    },

    /**
     * Calculates how many days to show from the next month
     * @return {number}
     */
    daysFromNextMonth() {
      const daysThisAndPrevMonth = this.daysFromPrevMonth + this.daysInMonth;
      return Math.ceil(daysThisAndPrevMonth / 7) * 7 - daysThisAndPrevMonth;
    },

    /**
     * Returns first-day-of-week as a number (Sunday is 0)
     * @return {Number}
     */
    firstDayOfWeekNumber() {
      return this.utils.getDayFromAbbr(this.firstDayOfWeek);
    },

    /**
     * The first day of the next page's month.
     * @return {Date}
     */
    firstOfNextMonth() {
      const d = new Date(this.pageDate);
      return new Date(this.utils.setMonth(d, this.utils.getMonth(d) + 1));
    },

    /**
     * A look-up object created from 'highlighted' prop
     * @return {Object}
     */
    highlightedConfig() {
      return new HighlightedDate(this.utils, this.disabledDates, this.highlighted).config;
    },

    /**
     * Is the next month disabled?
     * @return {Boolean}
     */
    isNextDisabled() {
      if (!this.disabledConfig.has.from) {
        return false;
      }

      return this.disabledConfig.from.month <= this.pageMonth && this.disabledConfig.from.year <= this.pageYear;
    },

    /**
     * Is the previous month disabled?
     * @return {Boolean}
     */
    isPreviousDisabled() {
      if (!this.disabledConfig.has.to) {
        return false;
      }

      return this.disabledConfig.to.month >= this.pageMonth && this.disabledConfig.to.year >= this.pageYear;
    },

    /**
     * Returns the current page's month as an integer.
     * @return {Number}
     */
    pageMonth() {
      return this.utils.getMonth(this.pageDate);
    },

    /**
     * Display the current page's month & year as the title.
     * @return {String}
     */
    pageTitleDay() {
      return this.translation.ymd ? `${this.currYearName} ${this.currMonthName}` : `${this.currMonthName} ${this.currYearName}`;
    }

  },
  methods: {
    /**
     * Set up a new date object to the first day of the current 'page'
     * @return Date
     */
    firstCellDate() {
      const d = this.pageDate;
      const firstOfMonth = this.useUtc ? new Date(Date.UTC(d.getUTCFullYear(), d.getUTCMonth(), 1)) : new Date(d.getFullYear(), d.getMonth(), 1, d.getHours(), d.getMinutes());
      return new Date(firstOfMonth.setDate(firstOfMonth.getDate() - this.daysFromPrevMonth));
    },

    /**
     * Whether a day is disabled
     * @param {Date} date to check if disabled
     * @return {Boolean}
     */
    isDisabledDate(date) {
      return new DisabledDate(this.utils, this.disabledDates).isDateDisabled(date);
    },

    /**
     * Whether a day is highlighted
     * (only if it is not disabled already except when highlighted.includeDisabled is true)
     * @param {Date} date to check if highlighted
     * @return {Boolean}
     */
    isHighlightedDate(date) {
      const dateWithoutTime = this.utils.resetDateTime(date);
      return new HighlightedDate(this.utils, this.disabledDates, this.highlighted).isDateHighlighted(dateWithoutTime);
    },

    /**
     * Whether a day is highlighted and it is the last date
     * in the highlighted range of dates
     * @param {Date} date end highlight
     * @return {Boolean}
     */
    isHighlightEnd(date) {
      const config = this.highlightedConfig;
      return this.isHighlightedDate(date) && config.to.year === this.utils.getFullYear(date) && config.to.month === this.utils.getMonth(date) && config.to.day === this.utils.getDate(date);
    },

    /**
     * Whether a day is highlighted and it is the first date
     * in the highlighted range of dates
     * @param {Date} date start highlight
     * @return {Boolean}
     */
    isHighlightStart(date) {
      const config = this.highlightedConfig;
      return this.isHighlightedDate(date) && config.from.year === this.utils.getFullYear(date) && config.from.month === this.utils.getMonth(date) && config.from.day === this.utils.getDate(date);
    },

    /**
     * Whether a day is selected
     * @param {Date} dObj to check if selected
     * @return {Boolean}
     */
    isSelectedDate(dObj) {
      return this.selectedDate && this.utils.compareDates(this.selectedDate, dObj);
    },

    /**
     * Defines the objects within the days array
     * @param  {id}  id
     * @param  {Date}  dObj
     * @return {Object}
     */
    // eslint-disable-next-line complexity
    makeDay(id, dObj) {
      const isNextMonth = dObj >= this.firstOfNextMonth;
      const isPreviousMonth = dObj < this.pageDate;
      const isSaturday = this.utils.getDay(dObj) === 6;
      const isSunday = this.utils.getDay(dObj) === 0;
      const showDate = this.showEdgeDates || !(isPreviousMonth || isNextMonth);
      return {
        date: showDate ? this.utils.getDate(dObj) : '',
        timestamp: dObj.valueOf(),
        isSelected: this.isSelectedDate(dObj),
        isDisabled: showDate ? this.isDisabledDate(dObj) : true,
        isHighlighted: this.isHighlightedDate(dObj),
        isHighlightStart: this.isHighlightStart(dObj),
        isHighlightEnd: this.isHighlightEnd(dObj),
        isOpenDate: this.openDate && this.utils.compareDates(dObj, this.openDate),
        isToday: this.utils.compareDates(dObj, new Date()),
        isWeekend: isSaturday || isSunday,
        isSaturday,
        isSunday,
        isPreviousMonth,
        isNextMonth
      };
    }

  }
};

/* script */
const __vue_script__$4 = script$4;

/* template */
var __vue_render__$3 = function () {
  var _vm = this;
  var _h = _vm.$createElement;
  var _c = _vm._self._c || _h;
  return _c(
    "div",
    [
      _vm.$slots.beforeCalendarHeaderDay
        ? _c("div", [_vm._t("beforeCalendarHeaderDay")], 2)
        : _vm._e(),
      _vm._v(" "),
      _vm.showHeader
        ? _c(
            "PickerHeader",
            {
              ref: "pickerHeader",
              attrs: {
                "bootstrap-styling": _vm.bootstrapStyling,
                "is-next-disabled": _vm.isNextDisabled,
                "is-previous-disabled": _vm.isPreviousDisabled,
                "is-rtl": _vm.isRtl,
              },
              on: {
                "focus-input": _vm.focusInput,
                "page-change": function ($event) {
                  return _vm.changePage($event)
                },
                "set-focus": function ($event) {
                  return _vm.$emit("set-focus", $event)
                },
              },
            },
            [
              _vm._t("prevIntervalBtn", null, { slot: "prevIntervalBtn" }),
              _vm._v(" "),
              _c(
                "UpButton",
                {
                  ref: "up",
                  class: { btn: _vm.bootstrapStyling },
                  attrs: {
                    "is-disabled": _vm.isUpDisabled,
                    "is-rtl": _vm.isRtl,
                  },
                  on: {
                    "focus-input": _vm.focusInput,
                    select: function ($event) {
                      return _vm.$emit("set-view", "month")
                    },
                    "set-focus": function ($event) {
                      return _vm.$emit("set-focus", $event)
                    },
                  },
                },
                [_vm._v("\n      " + _vm._s(_vm.pageTitleDay) + "\n    ")]
              ),
              _vm._v(" "),
              _vm._t("nextIntervalBtn", null, { slot: "nextIntervalBtn" }),
            ],
            2
          )
        : _vm._e(),
      _vm._v(" "),
      _c("div", [
        _c(
          "div",
          { staticClass: "day-header" },
          _vm._l(_vm.daysOfWeek, function (day) {
            return _c("span", { key: day }, [
              _vm._v("\n        " + _vm._s(day) + "\n      "),
            ])
          }),
          0
        ),
        _vm._v(" "),
        _c(
          "div",
          { staticClass: "cells-wrapper" },
          [
            _c(
              "Transition",
              { attrs: { name: _vm.transitionName } },
              [
                _c("PickerCells", {
                  key: _vm.pageTitleDay,
                  ref: "cells",
                  attrs: {
                    "bootstrap-styling": _vm.bootstrapStyling,
                    cells: _vm.cells,
                    "is-rtl": _vm.isRtl,
                    "show-edge-dates": _vm.showEdgeDates,
                    "tabbable-cell-id": _vm.tabbableCellId,
                    view: "day",
                  },
                  on: {
                    arrow: function ($event) {
                      return _vm.handleArrow($event)
                    },
                    select: function ($event) {
                      return _vm.select($event)
                    },
                  },
                  scopedSlots: _vm._u(
                    [
                      {
                        key: "default",
                        fn: function (ref) {
                          var cell = ref.cell;
                          return [
                            _vm._t(
                              "dayCellContent",
                              [
                                _vm._v(
                                  "\n            " +
                                    _vm._s(_vm.dayCellContent(cell)) +
                                    "\n          "
                                ),
                              ],
                              { cell: cell }
                            ),
                          ]
                        },
                      },
                    ],
                    null,
                    true
                  ),
                }),
              ],
              1
            ),
          ],
          1
        ),
      ]),
      _vm._v(" "),
      _vm.$slots.calendarFooterDay
        ? _c("div", [_vm._t("calendarFooterDay")], 2)
        : _vm._e(),
    ],
    1
  )
};
var __vue_staticRenderFns__$3 = [];
__vue_render__$3._withStripped = true;

  /* style */
  const __vue_inject_styles__$4 = undefined;
  /* scoped */
  const __vue_scope_id__$4 = undefined;
  /* module identifier */
  const __vue_module_identifier__$4 = undefined;
  /* functional template */
  const __vue_is_functional_template__$4 = false;
  /* style inject */
  
  /* style inject SSR */
  
  /* style inject shadow dom */
  

  
  const __vue_component__$4 = /*#__PURE__*/normalizeComponent(
    { render: __vue_render__$3, staticRenderFns: __vue_staticRenderFns__$3 },
    __vue_inject_styles__$4,
    __vue_script__$4,
    __vue_scope_id__$4,
    __vue_is_functional_template__$4,
    __vue_module_identifier__$4,
    false,
    undefined,
    undefined,
    undefined
  );

//
var script$3 = {
  name: 'PickerMonth',
  components: {
    PickerCells: __vue_component__$6,
    UpButton: __vue_component__$5
  },
  mixins: [__vue_component__$7],
  computed: {
    /**
     * Sets an array with all months to show this year
     * @return {Array}
     */
    cells() {
      const d = this.pageDate;
      const months = []; // set up a new date object to the beginning of the current 'page'

      const dObj = this.useUtc ? new Date(Date.UTC(d.getUTCFullYear(), 0, d.getUTCDate())) : new Date(d.getFullYear(), 0, d.getDate(), d.getHours(), d.getMinutes());
      const todayMonth = new Date(this.utils.setDate(this.utils.getNewDateObject(), 1));

      for (let i = 0; i < 12; i += 1) {
        months.push({
          month: this.utils.getMonthName(i, this.translation.months),
          timestamp: dObj.valueOf(),
          isDisabled: this.isDisabledMonth(dObj),
          isOpenDate: this.isOpenMonth(dObj),
          isSelected: this.isSelectedMonth(dObj),
          isToday: this.utils.compareDates(dObj, todayMonth)
        });
        this.utils.setMonth(dObj, this.utils.getMonth(dObj) + 1);
      }

      return months;
    },

    /**
     * Is the next year disabled?
     * @return {Boolean}
     */
    isNextDisabled() {
      if (!this.disabledConfig.has.from) {
        return false;
      }

      return this.disabledConfig.from.year <= this.pageYear;
    },

    /**
     * Is the previous year disabled?
     * @return {Boolean}
     */
    isPreviousDisabled() {
      if (!this.disabledConfig.has.to) {
        return false;
      }

      return this.disabledConfig.to.year >= this.pageYear;
    },

    /**
     * Display the current page's year as the title.
     * @return {String}
     */
    pageTitleMonth() {
      const {
        yearSuffix
      } = this.translation;
      return `${this.pageYear}${yearSuffix}`;
    }

  },
  methods: {
    /**
     * Whether a month is disabled
     * @param {Date} date
     * @return {Boolean}
     */
    isDisabledMonth(date) {
      return new DisabledDate(this.utils, this.disabledDates).isMonthDisabled(date);
    },

    /**
     * Should the calendar open on this month?
     * @return {Boolean}
     */
    isOpenMonth(date) {
      if (!this.openDate) {
        return false;
      }

      const openDateMonth = this.utils.getMonth(this.openDate);
      const openDateYear = this.utils.getFullYear(this.openDate);
      const thisDateMonth = this.utils.getMonth(date);
      const thisDateYear = this.utils.getFullYear(date);
      return openDateMonth === thisDateMonth && openDateYear === thisDateYear;
    },

    /**
     * Whether the selected date is in this month
     * @param {Date} date
     * @return {Boolean}
     */
    isSelectedMonth(date) {
      const month = this.utils.getMonth(date);
      const year = this.utils.getFullYear(date);
      return this.selectedDate && year === this.utils.getFullYear(this.selectedDate) && month === this.utils.getMonth(this.selectedDate);
    }

  }
};

/* script */
const __vue_script__$3 = script$3;

/* template */
var __vue_render__$2 = function () {
  var _vm = this;
  var _h = _vm.$createElement;
  var _c = _vm._self._c || _h;
  return _c(
    "div",
    [
      _vm.$slots.beforeCalendarHeaderMonth
        ? _c("div", [_vm._t("beforeCalendarHeaderMonth")], 2)
        : _vm._e(),
      _vm._v(" "),
      _vm.showHeader
        ? _c(
            "PickerHeader",
            {
              ref: "pickerHeader",
              attrs: {
                "bootstrap-styling": _vm.bootstrapStyling,
                "is-next-disabled": _vm.isNextDisabled,
                "is-previous-disabled": _vm.isPreviousDisabled,
                "is-rtl": _vm.isRtl,
              },
              on: {
                "focus-input": _vm.focusInput,
                "page-change": function ($event) {
                  return _vm.changePage($event)
                },
                "set-focus": function ($event) {
                  return _vm.$emit("set-focus", $event)
                },
              },
            },
            [
              _vm._t("prevIntervalBtn", null, { slot: "prevIntervalBtn" }),
              _vm._v(" "),
              _c(
                "UpButton",
                {
                  ref: "up",
                  class: { btn: _vm.bootstrapStyling },
                  attrs: {
                    "is-disabled": _vm.isUpDisabled,
                    "is-rtl": _vm.isRtl,
                  },
                  on: {
                    "focus-input": _vm.focusInput,
                    select: function ($event) {
                      return _vm.$emit("set-view", "year")
                    },
                    "set-focus": function ($event) {
                      return _vm.$emit("set-focus", $event)
                    },
                  },
                },
                [_vm._v("\n      " + _vm._s(_vm.pageTitleMonth) + "\n    ")]
              ),
              _vm._v(" "),
              _vm._t("nextIntervalBtn", null, { slot: "nextIntervalBtn" }),
            ],
            2
          )
        : _vm._e(),
      _vm._v(" "),
      _c(
        "div",
        { staticClass: "cells-wrapper" },
        [
          _c(
            "Transition",
            { attrs: { name: _vm.transitionName } },
            [
              _c("PickerCells", {
                key: _vm.pageTitleMonth,
                ref: "cells",
                attrs: {
                  "bootstrap-styling": _vm.bootstrapStyling,
                  cells: _vm.cells,
                  "is-rtl": _vm.isRtl,
                  "tabbable-cell-id": _vm.tabbableCellId,
                  view: "month",
                },
                on: {
                  arrow: function ($event) {
                    return _vm.handleArrow($event)
                  },
                  select: function ($event) {
                    return _vm.select($event)
                  },
                },
                scopedSlots: _vm._u([
                  {
                    key: "default",
                    fn: function (ref) {
                      var cell = ref.cell;
                      return [
                        _vm._v("\n        " + _vm._s(cell.month) + "\n      "),
                      ]
                    },
                  },
                ]),
              }),
            ],
            1
          ),
        ],
        1
      ),
      _vm._v(" "),
      _vm.$slots.calendarFooterMonth
        ? _c("div", [_vm._t("calendarFooterMonth")], 2)
        : _vm._e(),
    ],
    1
  )
};
var __vue_staticRenderFns__$2 = [];
__vue_render__$2._withStripped = true;

  /* style */
  const __vue_inject_styles__$3 = undefined;
  /* scoped */
  const __vue_scope_id__$3 = undefined;
  /* module identifier */
  const __vue_module_identifier__$3 = undefined;
  /* functional template */
  const __vue_is_functional_template__$3 = false;
  /* style inject */
  
  /* style inject SSR */
  
  /* style inject shadow dom */
  

  
  const __vue_component__$3 = /*#__PURE__*/normalizeComponent(
    { render: __vue_render__$2, staticRenderFns: __vue_staticRenderFns__$2 },
    __vue_inject_styles__$3,
    __vue_script__$3,
    __vue_scope_id__$3,
    __vue_is_functional_template__$3,
    __vue_module_identifier__$3,
    false,
    undefined,
    undefined,
    undefined
  );

//
var script$2 = {
  name: 'PickerYear',
  components: {
    PickerCells: __vue_component__$6,
    UpButton: __vue_component__$5
  },
  mixins: [__vue_component__$7],
  props: {
    yearRange: {
      type: Number,
      default: 10
    }
  },
  computed: {
    /**
     * Sets an array with all years to show this decade (or yearRange)
     * @return {Array}
     */
    // eslint-disable-next-line complexity,max-statements
    cells() {
      const d = this.pageDate;
      const years = [];
      const year = this.useUtc ? Math.floor(d.getUTCFullYear() / this.yearRange) * this.yearRange : Math.floor(d.getFullYear() / this.yearRange) * this.yearRange; // set up a new date object to the beginning of the current 'page'

      const dObj = this.useUtc ? new Date(Date.UTC(year, d.getUTCMonth(), d.getUTCDate())) : new Date(year, d.getMonth(), d.getDate(), d.getHours(), d.getMinutes());
      const todayYear = this.utils.getFullYear(this.utils.getNewDateObject());

      for (let i = 0; i < this.yearRange; i += 1) {
        years.push({
          year: this.utils.getFullYear(dObj),
          timestamp: dObj.valueOf(),
          isDisabled: this.isDisabledYear(dObj),
          isOpenDate: this.isOpenYear(dObj),
          isSelected: this.isSelectedYear(dObj),
          isToday: dObj.getFullYear() === todayYear
        });
        this.utils.setFullYear(dObj, this.utils.getFullYear(dObj) + 1);
      } // Fill any remaining cells with blanks to position trailing cells correctly when rtl


      const cellsInGrid = Math.ceil(this.yearRange / 3) * 3;

      for (let i = years.length; i < cellsInGrid; i += 1) {
        years.push({
          id: i,
          isDisabled: true
        });
      }

      return years;
    },

    /**
     * Is the next decade disabled?
     * @return {Boolean}
     */
    isNextDisabled() {
      if (!this.disabledConfig.has.from) {
        return false;
      }

      return this.disabledConfig.from.year <= this.pageDecadeEnd;
    },

    /**
     * Is the previous decade disabled?
     * @return {Boolean}
     */
    isPreviousDisabled() {
      if (!this.disabledConfig.has.to) {
        return false;
      }

      return this.disabledConfig.to.year >= this.pageDecadeStart;
    },

    /**
     * The year at which the current yearRange starts
     * @return {Number}
     */
    pageDecadeStart() {
      return Math.floor(this.pageYear / this.yearRange) * this.yearRange;
    },

    /**
     * The year at which the current yearRange ends
     * @return {Number}
     */
    pageDecadeEnd() {
      return this.pageDecadeStart + this.yearRange - 1;
    },

    /**
     * Display the current page's decade (or year range) as the title.
     * @return {String}
     */
    pageTitleYear() {
      const {
        yearSuffix
      } = this.translation;
      return `${this.pageDecadeStart} - ${this.pageDecadeEnd}${yearSuffix}`;
    }

  },
  methods: {
    /**
     * Whether a year is disabled
     * @param {Date} date
     * @return {Boolean}
     */
    isDisabledYear(date) {
      return new DisabledDate(this.utils, this.disabledDates).isYearDisabled(date);
    },

    /**
     * Should the calendar open on this year?
     * @return {Boolean}
     */
    isOpenYear(date) {
      if (!this.openDate) {
        return false;
      }

      const openDateYear = this.utils.getFullYear(this.openDate);
      const thisDateYear = this.utils.getFullYear(date);
      return openDateYear === thisDateYear;
    },

    /**
     * Whether the selected date is in this year
     * @param {Date} date
     * @return {Boolean}
     */
    isSelectedYear(date) {
      const year = this.utils.getFullYear(date);
      return this.selectedDate && year === this.utils.getFullYear(this.selectedDate);
    }

  }
};

/* script */
const __vue_script__$2 = script$2;

/* template */
var __vue_render__$1 = function () {
  var _vm = this;
  var _h = _vm.$createElement;
  var _c = _vm._self._c || _h;
  return _c(
    "div",
    [
      _vm.$slots.beforeCalendarHeaderYear
        ? _c("div", [_vm._t("beforeCalendarHeaderYear")], 2)
        : _vm._e(),
      _vm._v(" "),
      _vm.showHeader
        ? _c(
            "PickerHeader",
            {
              ref: "pickerHeader",
              attrs: {
                "bootstrap-styling": _vm.bootstrapStyling,
                "is-next-disabled": _vm.isNextDisabled,
                "is-previous-disabled": _vm.isPreviousDisabled,
                "is-rtl": _vm.isRtl,
              },
              on: {
                "focus-input": _vm.focusInput,
                "page-change": function ($event) {
                  return _vm.changePage($event)
                },
                "set-focus": function ($event) {
                  return _vm.$emit("set-focus", $event)
                },
              },
            },
            [
              _vm._t("prevIntervalBtn", null, { slot: "prevIntervalBtn" }),
              _vm._v(" "),
              _c(
                "UpButton",
                {
                  ref: "up",
                  class: { btn: _vm.bootstrapStyling },
                  attrs: { "is-disabled": true, "is-rtl": _vm.isRtl },
                },
                [_vm._v("\n      " + _vm._s(_vm.pageTitleYear) + "\n    ")]
              ),
              _vm._v(" "),
              _vm._t("nextIntervalBtn", null, { slot: "nextIntervalBtn" }),
            ],
            2
          )
        : _vm._e(),
      _vm._v(" "),
      _c(
        "div",
        { staticClass: "cells-wrapper" },
        [
          _c(
            "Transition",
            { attrs: { name: _vm.transitionName } },
            [
              _c("PickerCells", {
                key: _vm.pageTitleYear,
                ref: "cells",
                attrs: {
                  "bootstrap-styling": _vm.bootstrapStyling,
                  cells: _vm.cells,
                  "is-rtl": _vm.isRtl,
                  "tabbable-cell-id": _vm.tabbableCellId,
                  view: "year",
                },
                on: {
                  arrow: function ($event) {
                    return _vm.handleArrow($event)
                  },
                  select: function ($event) {
                    return _vm.select($event)
                  },
                },
                scopedSlots: _vm._u([
                  {
                    key: "default",
                    fn: function (ref) {
                      var cell = ref.cell;
                      return [
                        _vm._v("\n        " + _vm._s(cell.year) + "\n      "),
                      ]
                    },
                  },
                ]),
              }),
            ],
            1
          ),
        ],
        1
      ),
      _vm._v(" "),
      _vm.$slots.calendarFooterYear
        ? _c("div", [_vm._t("calendarFooterYear")], 2)
        : _vm._e(),
    ],
    1
  )
};
var __vue_staticRenderFns__$1 = [];
__vue_render__$1._withStripped = true;

  /* style */
  const __vue_inject_styles__$2 = undefined;
  /* scoped */
  const __vue_scope_id__$2 = undefined;
  /* module identifier */
  const __vue_module_identifier__$2 = undefined;
  /* functional template */
  const __vue_is_functional_template__$2 = false;
  /* style inject */
  
  /* style inject SSR */
  
  /* style inject shadow dom */
  

  
  const __vue_component__$2 = /*#__PURE__*/normalizeComponent(
    { render: __vue_render__$1, staticRenderFns: __vue_staticRenderFns__$1 },
    __vue_inject_styles__$2,
    __vue_script__$2,
    __vue_scope_id__$2,
    __vue_is_functional_template__$2,
    __vue_module_identifier__$2,
    false,
    undefined,
    undefined,
    undefined
  );

/* eslint no-param-reassign: 0 */

/**
 * get the hidden element width, height
 * @param {HTMLElement} element dom
 */
function getPopupElementSize(element) {
  const originalDisplay = element.style.display;
  const originalVisibility = element.style.visibility;
  element.style.display = 'block';
  element.style.visibility = 'hidden';
  const styles = window.getComputedStyle(element);
  const width = element.offsetWidth + parseInt(styles.marginLeft, 10) + parseInt(styles.marginRight, 10);
  const height = element.offsetHeight + parseInt(styles.marginTop, 10) + parseInt(styles.marginBottom, 10);
  element.style.display = originalDisplay;
  element.style.visibility = originalVisibility;
  return {
    width,
    height
  };
}
/**
 * get the popup position
 * @param {Element} el element
 * @param {Element} elRelative relative element
 * @param {Number} targetWidth target element's width
 * @param {Number} targetHeight target element's height
 * @param {Boolean} appendToBody
 * @param {String} fixedPosition
 * @param {Boolean} rtl
 */
// eslint-disable-next-line complexity,max-statements

function getRelativePosition({
  el,
  elRelative,
  targetWidth,
  targetHeight,
  appendToBody,
  fixedPosition,
  rtl
}) {
  let left = 0;
  let top = 0;
  let offsetX = 0;
  let offsetY = 0;
  const relativeRect = elRelative.getBoundingClientRect();
  const documentWidth = document.documentElement.clientWidth;
  const documentHeight = document.documentElement.clientHeight;

  if (appendToBody) {
    offsetX = window.pageXOffset + relativeRect.left;
    offsetY = window.pageYOffset + relativeRect.top;
  }

  const calendarBounding = el.getBoundingClientRect();
  const outOfBoundsRight = calendarBounding.right > window.innerWidth;
  const outOfBoundsBottom = calendarBounding.bottom > window.innerHeight;
  const fixedPositionRight = fixedPosition && fixedPosition.indexOf('right') !== -1;
  const fixedPositionTop = fixedPosition && fixedPosition.indexOf('top') !== -1;

  const setLeft = () => {
    left = offsetX;
  };

  const setRight = () => {
    left = offsetX + relativeRect.width - targetWidth;
  };

  const setBottom = () => {
    top = offsetY + relativeRect.height;
  };

  const setTop = () => {
    top = offsetY - targetHeight;
  };

  if (fixedPosition === '') {
    if (outOfBoundsRight || rtl) {
      setRight();
    } else {
      setLeft();
    }

    if (outOfBoundsBottom) {
      setTop();
    } else {
      setBottom();
    }

    const hasRelativeWidth = documentWidth - relativeRect.left < targetWidth && relativeRect.right < targetWidth;
    const hasRelativeHeight = relativeRect.top <= targetHeight && documentHeight - relativeRect.bottom <= targetHeight;

    if (hasRelativeWidth) {
      left = offsetX - relativeRect.left + 1;
    }

    if (hasRelativeHeight) {
      top = offsetY + documentHeight - relativeRect.top - targetHeight;
    }
  } else {
    if (fixedPositionRight) {
      setRight();
    } else {
      setLeft();
    }

    if (fixedPositionTop) {
      setTop();
    } else {
      setBottom();
    }
  }

  return {
    left: `${left}px`,
    top: `${top}px`
  };
}

var script$1 = {
  name: 'Popup',
  props: {
    appendToBody: {
      type: Boolean,
      default: true
    },
    fixedPosition: {
      type: String,
      default: ''
    },
    inline: {
      type: Boolean,
      default: false
    },
    rtl: {
      type: Boolean,
      default: false
    },
    visible: {
      type: Boolean,
      default: false
    }
  },

  data() {
    return {
      popupRect: null
    };
  },

  watch: {
    visible: {
      immediate: true,

      handler(val) {
        if (val) {
          this.displayPopup();
        }
      }

    }
  },

  mounted() {
    if (this.inline) {
      return;
    }

    if (this.appendToBody) {
      document.body.appendChild(this.$el);
    }
  },

  beforeDestroy() {
    if (this.inline) {
      return;
    }

    if (this.appendToBody && this.$el.parentNode) {
      this.$el.parentNode.removeChild(this.$el);
    }
  },

  methods: {
    /**
     * Adjusts the popup's `top` style attribute when `append-to-body` is true
     */
    setTopStyle() {
      if (this.appendToBody) {
        const relativeRect = this.$parent.$el.getBoundingClientRect();
        this.$el.style.top = `${relativeRect.bottom + window.scrollY}px`;
      }
    },

    /**
     * Sets the `left` and `top` style attributes of the popup
     */
    displayPopup() {
      if (this.inline || !this.visible) return;
      this.setTopStyle();
      const popup = this.$el;
      const relativeElement = this.$parent.$el;

      if (!this.popupRect) {
        this.popupRect = getPopupElementSize(popup);
      }

      const {
        width,
        height
      } = this.popupRect;
      const {
        left,
        top
      } = getRelativePosition({
        el: popup,
        elRelative: relativeElement,
        targetWidth: width,
        targetHeight: height,
        appendToBody: this.appendToBody,
        fixedPosition: this.fixedPosition,
        rtl: this.rtl
      });
      this.$el.style.left = left;
      this.$el.style.top = top;
    }

  },

  render() {
    return this.$slots.default;
  }

};

/* script */
const __vue_script__$1 = script$1;

/* template */

  /* style */
  const __vue_inject_styles__$1 = undefined;
  /* scoped */
  const __vue_scope_id__$1 = undefined;
  /* module identifier */
  const __vue_module_identifier__$1 = undefined;
  /* functional template */
  const __vue_is_functional_template__$1 = undefined;
  /* style inject */
  
  /* style inject SSR */
  
  /* style inject shadow dom */
  

  
  const __vue_component__$1 = /*#__PURE__*/normalizeComponent(
    {},
    __vue_inject_styles__$1,
    __vue_script__$1,
    __vue_scope_id__$1,
    __vue_is_functional_template__$1,
    __vue_module_identifier__$1,
    false,
    undefined,
    undefined,
    undefined
  );

//
var script = {
  name: 'Datepicker',
  components: {
    DateInput: __vue_component__$a,
    PickerDay: __vue_component__$4,
    PickerMonth: __vue_component__$3,
    PickerYear: __vue_component__$2,
    Popup: __vue_component__$1
  },
  mixins: [__vue_component__$b, __vue_component__$9],
  props: {
    appendToBody: {
      type: Boolean,
      default: false
    },
    calendarClass: {
      type: [String, Object, Array],
      default: ''
    },
    dayCellContent: {
      type: Function,
      default: day => day.date
    },
    disabledDates: {
      type: Object,

      default() {
        return {};
      }

    },
    firstDayOfWeek: {
      type: String,
      default: 'sun'
    },
    fixedPosition: {
      type: String,
      default: '',
      validator: val => {
        const possibleValues = ['', 'bottom', 'bottom-left', 'bottom-right', 'top', 'top-left', 'top-right'];
        return possibleValues.includes(val);
      }
    },
    fullMonthName: {
      type: Boolean,
      default: false
    },
    highlighted: {
      type: Object,

      default() {
        return {};
      }

    },
    initialView: {
      type: String,
      default: ''
    },
    language: {
      type: Object,
      default: () => en
    },
    maximumView: {
      type: String,
      default: 'year'
    },
    minimumView: {
      type: String,
      default: 'day'
    },
    showEdgeDates: {
      type: Boolean,
      default: true
    },
    showHeader: {
      type: Boolean,
      default: true
    },
    value: {
      type: [String, Date, Number],
      default: '',
      validator: val => val === null || val instanceof Date || typeof val === 'string' || typeof val === 'number'
    },
    wrapperClass: {
      type: [String, Object, Array],
      default: ''
    },
    yearPickerRange: {
      type: Number,
      default: 10
    }
  },

  data() {
    const utils = makeDateUtils(this.useUtc);
    const startDate = utils.getNewDateObject(this.openDate || null);
    const pageTimestamp = utils.setDate(startDate, 1);
    return {
      calendarHeight: 0,
      calendarSlots,
      isClickOutside: false,

      /*
       * Vue cannot observe changes to a Date Object so date must be stored as a timestamp
       * This represents the first day of the current viewing month
       * {Number}
       */
      pageTimestamp,

      /*
       * Selected Date
       * {Date}
       */
      selectedDate: null,
      slideDuration: 250,
      utils,
      view: ''
    };
  },

  computed: {
    computedInitialView() {
      return this.initialView || this.minimumView;
    },

    computedOpenDate() {
      const openDateOrToday = this.openDate ? new Date(this.openDate) : this.utils.getNewDateObject();
      const openDate = this.selectedDate || openDateOrToday; // If the `minimum-view` is `month` or `year`, convert `openDate` accordingly

      return this.minimumView === 'day' ? openDate : new Date(this.utils.setDate(openDate, 1));
    },

    datepickerId() {
      return `vdp-${Math.random().toString(36).slice(-10)}`;
    },

    isInline() {
      return !!this.inline;
    },

    isOpen() {
      return this.view !== '';
    },

    isMinimumView() {
      return this.view === this.minimumView;
    },

    isRtl() {
      return this.translation.rtl;
    },

    isUpDisabled() {
      return !this.allowedToShowView(this.nextView.up);
    },

    nextView() {
      const views = ['day', 'month', 'year'];

      const isCurrentView = view => view === this.view;

      const viewIndex = views.findIndex(isCurrentView);

      const nextViewDown = index => {
        return index <= 0 ? undefined : views[index - 1];
      };

      const nextViewUp = index => {
        if (index < 0) {
          return undefined;
        }

        if (index === views.length - 1) {
          return 'decade';
        }

        return views[index + 1];
      };

      return {
        up: nextViewUp(viewIndex),
        down: nextViewDown(viewIndex)
      };
    },

    pageDate() {
      return new Date(this.pageTimestamp);
    },

    picker() {
      const view = this.view || this.computedInitialView;
      return `Picker${this.ucFirst(view)}`;
    },

    pickerClasses() {
      return [this.calendarClass, this.isInline && 'vdp-datepicker__calendar--inline', this.isRtl && this.appendToBody && 'rtl'];
    },

    translation() {
      return this.language;
    }

  },
  watch: {
    initialView() {
      if (this.isOpen) {
        this.setInitialView();
      }
    },

    isActive(hasJustBecomeActive) {
      if (hasJustBecomeActive && this.inline) {
        this.setNavElementsFocusedIndex();
        this.tabToCorrectInlineCell();
      }
    },

    openDate() {
      this.setPageDate();
    },

    value(value) {
      const parsedValue = this.parseValue(value);
      this.setValue(parsedValue);
    },

    view(newView, oldView) {
      this.handleViewChange(newView, oldView);
    }

  },

  mounted() {
    this.init();
    document.addEventListener('click', this.handleClickOutside);
  },

  beforeDestroy() {
    document.removeEventListener('click', this.handleClickOutside);
  },

  methods: {
    /**
     * Are we allowed to show a specific picker view?
     * @param {String} view
     * @return {Boolean}
     */
    allowedToShowView(view) {
      const views = ['day', 'month', 'year'];
      const minimumViewIndex = views.indexOf(this.minimumView);
      const maximumViewIndex = views.indexOf(this.maximumView);
      const viewIndex = views.indexOf(view);
      return viewIndex >= minimumViewIndex && viewIndex <= maximumViewIndex;
    },

    /**
     * Clear the selected date
     */
    clearDate() {
      if (!this.selectedDate) {
        return;
      }

      this.selectedDate = null;
      this.focus.refs = ['input'];
      this.close();
      this.setPageDate();
      this.$emit('selected', null);
      this.$emit('input', null);
      this.$emit('cleared');
    },

    /**
     * Close the calendar
     */
    close() {
      if (this.isInline) {
        return;
      }

      this.view = '';

      if (this.showCalendarOnFocus) {
        this.$refs.dateInput.shouldToggleOnClick = true;
      }

      if (this.isClickOutside) {
        this.isClickOutside = false;
      } else {
        this.reviewFocus();
      }

      this.$emit('closed');
    },

    /**
     * Closes the calendar when no element within it has focus
     */
    handleClickOutside() {
      if (document.datepickerId !== this.datepickerId) {
        return;
      }

      const isFocused = this.allElements.includes(document.activeElement);

      if (!isFocused && this.isOpen) {
        this.isClickOutside = true;
        this.close();
      }
    },

    /**
     * Emits a 'blur' event
     */
    handleInputBlur() {
      this.$emit('blur');
    },

    /**
     * Emits a 'focus' event
     */
    handleInputFocus() {
      this.$emit('focus');
    },

    /**
     * Set the new pageDate, focus the relevant element and emit a `changed-<view>` event
     */
    handlePageChange({
      focusRefs,
      pageDate
    }) {
      this.setPageDate(pageDate);
      this.focus.refs = focusRefs;
      this.focus.delay = this.slideDuration;
      this.reviewFocus();
      this.$emit(`changed-${this.nextView.up}`, pageDate);
    },

    /**
     * Set the date, or go to the next view down
     */
    // eslint-disable-next-line max-statements,complexity
    handleSelect(cell) {
      if (this.allowedToShowView(this.nextView.down)) {
        this.showNextViewDown(cell);
        return;
      }

      this.$refs.dateInput.typedDate = '';
      this.selectDate(cell.timestamp);
      this.focus.delay = cell.isNextMonth ? this.slideDuration : 0;
      this.focus.refs = this.isInline ? ['tabbableCell'] : ['input'];
      this.close();

      if (this.showCalendarOnFocus && !this.inline) {
        this.$refs.dateInput.shouldToggleOnClick = true;
      } else {
        this.reviewFocus();
      }
    },

    /**
     * Set the date from a 'typed-date' event
     * @param {Date} date
     */
    handleTypedDate(date) {
      if (this.selectedDate) {
        this.setTransitionAndFocusDelay(this.selectedDate, date);
      }

      this.selectDate(date ? date.valueOf() : null);
      this.reviewFocus();
    },

    /**
     * Focus the relevant element when the view changes
     * @param {String} newView
     * @param {String} oldView
     */
    handleViewChange(newView, oldView) {
      const isClosing = newView === '';
      const isOpeningInline = oldView === '' && this.isInline;

      if (isClosing || isOpeningInline) {
        return;
      }

      if (!this.isRevertingToOpenDate) {
        this.setViewChangeFocusRefs(newView, oldView);
        this.reviewFocus();
      }

      this.isRevertingToOpenDate = false;
    },

    /**
     * Returns true if element has the given className
     * @param   {HTMLElement} element
     * @param   {String}      className
     * @returns {Boolean}
     */
    hasClass(element, className) {
      return element && element.className.split(' ').includes(className);
    },

    /**
     * Initiate the component
     */
    init() {
      if (this.value) {
        let parsedValue = this.parseValue(this.value);
        const isDateDisabled = parsedValue && this.isDateDisabled(parsedValue);

        if (isDateDisabled) {
          parsedValue = null;
          this.$emit('input', parsedValue);
        }

        this.setValue(parsedValue);
      }

      if (this.isInline) {
        this.setInitialView();
      }

      this.setSlideDuration();
    },

    /**
     * Returns true if a date is disabled
     * @param {Date} date
     */
    isDateDisabled(date) {
      return new DisabledDate(this.utils, this.disabledDates).isDateDisabled(date);
    },

    /**
     * Returns true if we should reset the focus to computedOpenDate
     * @returns {Boolean}
     */
    isResetFocus() {
      if (!this.isOpen) {
        return false;
      }

      const isOpenCellFocused = this.hasClass(document.activeElement, 'cell') && !this.hasClass(document.activeElement, 'open');
      return !this.isMinimumView || isOpenCellFocused;
    },

    /**
     * Opens the calendar with the relevant view: 'day', 'month', or 'year'
     */
    open() {
      if (this.disabled || this.isInline) {
        return;
      }

      this.setInitialView();
      this.reviewFocus();
      this.$emit('opened');
    },

    /**
     * Parse a datepicker value from string/number to date
     * @param   {Date|String|Number|null} date
     * @returns {Date}
     */
    parseValue(date) {
      let dateTemp = date;

      if (typeof dateTemp === 'string' || typeof dateTemp === 'number') {
        const parsed = new Date(dateTemp);
        dateTemp = Number.isNaN(parsed.valueOf()) ? null : parsed;
      }

      return dateTemp;
    },

    /**
     * Focus the open date, or close the calendar if already focused
     */
    resetOrClose() {
      if (this.isResetFocus()) {
        this.resetFocusToOpenDate();
        return;
      }

      if (this.isOpen) {
        this.focus.refs = ['input'];
        this.close();
      }
    },

    /**
     * Select the date
     * @param {Number} timestamp
     */
    selectDate(timestamp) {
      if (!timestamp) {
        this.selectedDate = null;
        return;
      }

      const date = new Date(timestamp);
      this.selectedDate = date;
      this.setPageDate(date);
      this.$emit('selected', date);
      this.$emit('input', date);
    },

    /**
     * Sets the initial picker page view: day, month or year
     */
    setInitialView() {
      const initialView = this.computedInitialView;

      if (!this.allowedToShowView(initialView)) {
        throw new Error(`initialView '${this.initialView}' cannot be rendered based on minimum '${this.minimumView}' and maximum '${this.maximumView}'`);
      }

      this.setView(initialView);
    },

    /**
     * Sets the date that the calendar should open on
     */
    setPageDate(date) {
      let dateTemp = date;

      if (!dateTemp) {
        if (this.computedOpenDate) {
          dateTemp = new Date(this.computedOpenDate);
        } else {
          dateTemp = new Date();
        }

        dateTemp = this.utils.resetDateTime(dateTemp);
      }

      this.pageTimestamp = this.utils.setDate(new Date(dateTemp), 1);
    },

    /**
     * Sets the slide duration in milliseconds by looking up the stylesheet
     */
    setSlideDuration() {
      if (!this.$refs.picker || !this.$refs.picker.$refs.cells) {
        return;
      }

      const cells = this.$refs.picker.$refs.cells.$el;
      const durationInSecs = window.getComputedStyle(cells).transitionDuration;
      this.slideDuration = parseFloat(durationInSecs) * 1000;
    },

    /**
     * Set the datepicker value
     * @param {Date|String|Number|null} date
     */
    setValue(date) {
      if (!date) {
        this.selectedDate = null;
        this.setPageDate();
        return;
      }

      this.selectedDate = date;
      this.setPageDate(date);
    },

    /**
     * Set the picker view
     * @param {String} view
     */
    setView(view) {
      if (this.allowedToShowView(view)) {
        this.view = view;
      }
    },

    /**
     * Sets the array of `refs` that might be focused following a view change
     * @param {String} newView The view being changed to
     * @param {String} oldView The previous view
     */
    setViewChangeFocusRefs(newView, oldView) {
      if (oldView === '') {
        this.focus.refs = [];
        return;
      }

      const views = ['day', 'month', 'year'];

      const isNewView = view => view === newView;

      const isOldView = view => view === oldView;

      const newViewIndex = views.findIndex(isNewView);
      const oldViewIndex = views.findIndex(isOldView);
      const isViewChangeUp = newViewIndex - oldViewIndex > 0;
      this.focus.refs = isViewChangeUp ? ['up', 'tabbableCell'] : ['tabbableCell', 'up'];
    },

    /**
     * Set the view to the next view down e.g. from `month` to `day`
     * @param {Object} cell The currently focused cell
     */
    showNextViewDown(cell) {
      this.setPageDate(new Date(cell.timestamp));
      this.$emit(`changed-${this.view}`, cell);
      this.setView(this.nextView.down);
    },

    /**
     * Capitalizes the first letter
     * @param {String} str The string to capitalize
     * @returns {String}
     */
    ucFirst(str) {
      return str[0].toUpperCase() + str.substring(1);
    }

  }
};

/* script */
const __vue_script__ = script;
/* template */
var __vue_render__ = function () {
  var _vm = this;
  var _h = _vm.$createElement;
  var _c = _vm._self._c || _h;
  return _c(
    "div",
    {
      ref: "datepicker",
      staticClass: "vdp-datepicker",
      class: [_vm.wrapperClass, { rtl: _vm.isRtl }],
      attrs: { id: _vm.datepickerId },
      on: {
        focusin: function ($event) {
          return _vm.handleFocusIn($event)
        },
        focusout: function ($event) {
          return _vm.handleFocusOut($event)
        },
        keydown: [
          function ($event) {
            if (
              !$event.type.indexOf("key") &&
              _vm._k($event.keyCode, "esc", 27, $event.key, ["Esc", "Escape"])
            ) {
              return null
            }
            return _vm.resetOrClose($event)
          },
          function ($event) {
            if (
              !$event.type.indexOf("key") &&
              _vm._k($event.keyCode, "tab", 9, $event.key, "Tab")
            ) {
              return null
            }
            return _vm.tabThroughNavigation($event)
          },
        ],
      },
    },
    [
      _c(
        "DateInput",
        {
          ref: "dateInput",
          attrs: {
            id: _vm.id,
            autofocus: _vm.autofocus,
            "bootstrap-styling": _vm.bootstrapStyling,
            "calendar-button": _vm.calendarButton,
            "calendar-button-icon": _vm.calendarButtonIcon,
            "calendar-button-icon-content": _vm.calendarButtonIconContent,
            "clear-button": _vm.clearButton,
            "clear-button-icon": _vm.clearButtonIcon,
            disabled: _vm.disabled,
            format: _vm.format,
            inline: _vm.inline,
            "input-class": _vm.inputClass,
            "is-open": _vm.isOpen,
            maxlength: _vm.maxlength,
            name: _vm.name,
            parser: _vm.parser,
            pattern: _vm.pattern,
            placeholder: _vm.placeholder,
            "ref-name": _vm.refName,
            required: _vm.required,
            "selected-date": _vm.selectedDate,
            "show-calendar-on-button-click": _vm.showCalendarOnButtonClick,
            "show-calendar-on-focus": _vm.showCalendarOnFocus,
            tabindex: _vm.tabindex,
            translation: _vm.translation,
            typeable: _vm.typeable,
            "use-utc": _vm.useUtc,
          },
          on: {
            blur: _vm.handleInputBlur,
            "clear-date": _vm.clearDate,
            close: _vm.close,
            focus: _vm.handleInputFocus,
            open: _vm.open,
            "set-focus": function ($event) {
              return _vm.setFocus($event)
            },
            "typed-date": _vm.handleTypedDate,
          },
        },
        [
          _vm._t("beforeDateInput", null, { slot: "beforeDateInput" }),
          _vm._v(" "),
          _vm._t("afterDateInput", null, { slot: "afterDateInput" }),
          _vm._v(" "),
          _vm._t("clearBtn", null, { slot: "clearBtn" }),
          _vm._v(" "),
          _vm._t("calendarBtn", null, { slot: "calendarBtn" }),
        ],
        2
      ),
      _vm._v(" "),
      _c(
        "Popup",
        {
          ref: "popup",
          attrs: {
            "append-to-body": _vm.appendToBody,
            "fixed-position": _vm.fixedPosition,
            inline: _vm.inline,
            rtl: _vm.isRtl,
            visible: _vm.isOpen,
          },
        },
        [
          _c("Transition", { attrs: { name: "toggle" } }, [
            _c(
              "div",
              {
                directives: [
                  {
                    name: "show",
                    rawName: "v-show",
                    value: _vm.isOpen,
                    expression: "isOpen",
                  },
                ],
                staticClass: "vdp-datepicker__calendar",
                class: _vm.pickerClasses,
                attrs: { "data-test-calendar": "" },
                on: {
                  mousedown: function ($event) {
                    $event.preventDefault();
                  },
                  focusin: function ($event) {
                    $event.stopPropagation();
                    return _vm.handleFocusIn($event)
                  },
                  focusout: function ($event) {
                    $event.stopPropagation();
                    return _vm.handleFocusOut($event)
                  },
                  keydown: [
                    function ($event) {
                      if (
                        !$event.type.indexOf("key") &&
                        _vm._k($event.keyCode, "esc", 27, $event.key, [
                          "Esc",
                          "Escape",
                        ])
                      ) {
                        return null
                      }
                      $event.stopPropagation();
                      return _vm.resetOrClose($event)
                    },
                    function ($event) {
                      if (
                        !$event.type.indexOf("key") &&
                        _vm._k($event.keyCode, "tab", 9, $event.key, "Tab")
                      ) {
                        return null
                      }
                      $event.stopPropagation();
                      return _vm.tabThroughNavigation($event)
                    },
                  ],
                },
              },
              [
                _c("Transition", { attrs: { name: "view" } }, [
                  _c(
                    "div",
                    { key: _vm.view, ref: "view" },
                    [
                      _vm.$slots.beforeCalendarHeader
                        ? _c("div", [_vm._t("beforeCalendarHeader")], 2)
                        : _vm._e(),
                      _vm._v(" "),
                      _c(
                        _vm.picker,
                        {
                          ref: "picker",
                          tag: "Component",
                          staticClass: "picker-view",
                          attrs: {
                            "bootstrap-styling": _vm.bootstrapStyling,
                            "day-cell-content": _vm.dayCellContent,
                            "disabled-dates": _vm.disabledDates,
                            "first-day-of-week": _vm.firstDayOfWeek,
                            highlighted: _vm.highlighted,
                            "is-rtl": _vm.isRtl,
                            "is-typeable": _vm.typeable,
                            "is-up-disabled": _vm.isUpDisabled,
                            "is-minimum-view": _vm.isMinimumView,
                            "open-date": _vm.computedOpenDate,
                            "page-date": _vm.pageDate,
                            "selected-date": _vm.selectedDate,
                            "show-edge-dates": _vm.showEdgeDates,
                            "show-full-month-name": _vm.fullMonthName,
                            "show-header": _vm.showHeader,
                            "slide-duration": _vm.slideDuration,
                            "tabbable-cell-id": _vm.tabbableCellId,
                            "transition-name": _vm.transitionName,
                            translation: _vm.translation,
                            "use-utc": _vm.useUtc,
                            view: _vm.view || _vm.computedInitialView,
                            "year-range": _vm.yearPickerRange,
                          },
                          on: {
                            "page-change": _vm.handlePageChange,
                            select: _vm.handleSelect,
                            "set-focus": function ($event) {
                              return _vm.setFocus($event)
                            },
                            "set-transition-name": function ($event) {
                              return _vm.setTransitionName($event)
                            },
                            "set-view": _vm.setView,
                          },
                          scopedSlots: _vm._u(
                            [
                              {
                                key: "dayCellContent",
                                fn: function (ref) {
                                  var cell = ref.cell;
                                  return [
                                    cell
                                      ? _vm._t("dayCellContent", null, {
                                          cell: cell,
                                        })
                                      : _vm._e(),
                                  ]
                                },
                              },
                            ],
                            null,
                            true
                          ),
                        },
                        [
                          _vm._l(_vm.calendarSlots, function (slotKey) {
                            return [_vm._t(slotKey, null, { slot: slotKey })]
                          }),
                        ],
                        2
                      ),
                      _vm._v(" "),
                      _vm.$slots.calendarFooter
                        ? _c("div", [_vm._t("calendarFooter")], 2)
                        : _vm._e(),
                    ],
                    1
                  ),
                ]),
              ],
              1
            ),
          ]),
        ],
        1
      ),
    ],
    1
  )
};
var __vue_staticRenderFns__ = [];
__vue_render__._withStripped = true;

  /* style */
  const __vue_inject_styles__ = undefined;
  /* scoped */
  const __vue_scope_id__ = undefined;
  /* module identifier */
  const __vue_module_identifier__ = undefined;
  /* functional template */
  const __vue_is_functional_template__ = false;
  /* style inject */
  
  /* style inject SSR */
  
  /* style inject shadow dom */
  

  
  const __vue_component__ = /*#__PURE__*/normalizeComponent(
    { render: __vue_render__, staticRenderFns: __vue_staticRenderFns__ },
    __vue_inject_styles__,
    __vue_script__,
    __vue_scope_id__,
    __vue_is_functional_template__,
    __vue_module_identifier__,
    false,
    undefined,
    undefined,
    undefined
  );

export default __vue_component__;
//# sourceMappingURL=Datepicker.esm.js.map
